"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.upsertQuery = exports.updateInputTimerange = exports.updateInputFullScreen = exports.toggleLockTimeline = exports.toggleLockSocTrends = exports.setIsInspected = exports.removeInputLink = exports.isQueryInput = exports.deleteOneQuery = exports.addInputLink = void 0;
var _fp = require("lodash/fp");
var _get_time_range = require("../../utils/get_time_range");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateInputFullScreen = (inputId, fullScreen, state) => ({
  ...state,
  global: {
    ...state.global,
    fullScreen: inputId === _constants.InputsModelId.global ? fullScreen : state.global.fullScreen
  },
  timeline: {
    ...state.timeline,
    fullScreen: inputId === _constants.InputsModelId.timeline ? fullScreen : state.timeline.fullScreen
  }
});
exports.updateInputFullScreen = updateInputFullScreen;
const getTimeRange = (timerange, inputId, linkToId) => {
  if ((inputId === _constants.InputsModelId.global || inputId === _constants.InputsModelId.timeline) && linkToId === _constants.InputsModelId.socTrends) {
    return (0, _get_time_range.getPreviousTimeRange)(timerange);
  }
  if (inputId === _constants.InputsModelId.socTrends && (linkToId === _constants.InputsModelId.global || linkToId === _constants.InputsModelId.timeline)) {
    return (0, _get_time_range.getFutureTimeRange)(timerange);
  }
  return timerange;
};
const updateInputTimerange = (inputId, timerange, state) => {
  const input = (0, _fp.get)(inputId, state);
  if (input != null) {
    return {
      ...[inputId, ...input.linkTo].reduce((acc, linkToId) => ({
        ...acc,
        [linkToId]: {
          ...(0, _fp.get)(linkToId, state),
          timerange: getTimeRange(timerange, inputId, linkToId)
        }
      }), inputId === _constants.InputsModelId.timeline ? {
        ...state,
        global: {
          ...state.global,
          linkTo: []
        }
      } : state)
    };
  }
  return state;
};
exports.updateInputTimerange = updateInputTimerange;
const toggleLockTimeline = state => {
  const linkToIdAlreadyExist = state.global.linkTo.indexOf(_constants.InputsModelId.timeline);
  return linkToIdAlreadyExist > -1 ? removeInputLink([_constants.InputsModelId.global, _constants.InputsModelId.timeline], state) : addInputLink([_constants.InputsModelId.global, _constants.InputsModelId.timeline], state);
};
exports.toggleLockTimeline = toggleLockTimeline;
const toggleLockSocTrends = state => {
  const linkToIdAlreadyExist = state.global.linkTo.indexOf(_constants.InputsModelId.socTrends);
  return linkToIdAlreadyExist > -1 ? removeInputLink([_constants.InputsModelId.global, _constants.InputsModelId.socTrends], state) : addInputLink([_constants.InputsModelId.global, _constants.InputsModelId.socTrends], state);
};
exports.toggleLockSocTrends = toggleLockSocTrends;
const upsertQuery = ({
  inputId,
  id,
  inspect,
  loading,
  refetch,
  state,
  searchSessionId
}) => {
  const queryIndex = state[inputId].queries.findIndex(q => q.id === id);
  return {
    ...state,
    [inputId]: {
      ...(0, _fp.get)(inputId, state),
      queries: queryIndex > -1 ? [...state[inputId].queries.slice(0, queryIndex), {
        id,
        inspect,
        isInspected: state[inputId].queries[queryIndex].isInspected,
        loading,
        refetch,
        searchSessionId: state[inputId].queries[queryIndex].searchSessionId,
        selectedInspectIndex: state[inputId].queries[queryIndex].selectedInspectIndex
      }, ...state[inputId].queries.slice(queryIndex + 1)] : [...state[inputId].queries, {
        id,
        inspect,
        isInspected: false,
        loading,
        refetch,
        selectedInspectIndex: 0,
        searchSessionId
      }]
    }
  };
};
exports.upsertQuery = upsertQuery;
const setIsInspected = ({
  id,
  inputId,
  isInspected,
  selectedInspectIndex,
  state,
  searchSessionId
}) => {
  const myQueryIndex = state[inputId].queries.findIndex(q => q.id === id);
  const myQuery = myQueryIndex > -1 ? state[inputId].queries[myQueryIndex] : null;
  return {
    ...state,
    [inputId]: {
      ...(0, _fp.get)(inputId, state),
      queries: myQueryIndex > -1 ? [...state[inputId].queries.slice(0, myQueryIndex), {
        ...myQuery,
        isInspected,
        selectedInspectIndex,
        searchSessionId
      }, ...state[inputId].queries.slice(myQueryIndex + 1)] : [...state[inputId].queries]
    }
  };
};
exports.setIsInspected = setIsInspected;
const addInputLink = (linkToIds, state) => {
  if (linkToIds.length !== 2) {
    throw new Error('Only link 2 input states at a time');
  }
  if (linkToIds.includes(_constants.InputsModelId.socTrends) && linkToIds.includes(_constants.InputsModelId.timeline)) {
    throw new Error('Do not link socTrends to timeline. Only link socTrends to global');
  }
  if (Array.from(new Set(linkToIds)).length === 1) {
    throw new Error('Input linkTo cannot link to itself');
  }
  if (linkToIds.includes(_constants.InputsModelId.timeline) && linkToIds.includes(_constants.InputsModelId.global)) {
    const socTrends = state.timeline.linkTo.includes(_constants.InputsModelId.socTrends) || state.global.linkTo.includes(_constants.InputsModelId.socTrends) ? [_constants.InputsModelId.socTrends] : [];
    return {
      ...state,
      timeline: {
        ...state.timeline,
        linkTo: [...socTrends, _constants.InputsModelId.global]
      },
      global: {
        ...state.global,
        linkTo: [...socTrends, _constants.InputsModelId.timeline]
      },
      // TODO: remove state.socTrends check when socTrendsEnabled feature flag removed
      ...(state.socTrends && socTrends.length ? {
        socTrends: {
          ...state.socTrends,
          linkTo: [_constants.InputsModelId.global, _constants.InputsModelId.timeline]
        }
      } : {})
    };
  }
  if (linkToIds.includes(_constants.InputsModelId.socTrends) && linkToIds.includes(_constants.InputsModelId.global)) {
    const timeline = state.global.linkTo.includes(_constants.InputsModelId.timeline) ? [_constants.InputsModelId.timeline] : [];
    return {
      ...state,
      // TODO: remove state.socTrends check when socTrendsEnabled feature flag removed
      ...(state.socTrends ? {
        socTrends: {
          ...state.socTrends,
          linkTo: [...timeline, _constants.InputsModelId.global]
        }
      } : {}),
      global: {
        ...state.global,
        linkTo: [...timeline, _constants.InputsModelId.socTrends]
      },
      ...(timeline.length ? {
        timeline: {
          ...state.timeline,
          linkTo: [_constants.InputsModelId.global, _constants.InputsModelId.socTrends]
        }
      } : {})
    };
  }
  return state;
};
exports.addInputLink = addInputLink;
const removeInputLink = (linkToIds, state) => {
  if (linkToIds.length !== 2) {
    throw new Error('Only remove linkTo from 2 input states at a time');
  }
  if (linkToIds.includes(_constants.InputsModelId.socTrends) && linkToIds.includes(_constants.InputsModelId.timeline)) {
    throw new Error('Do not remove link socTrends to timeline. Only remove link socTrends to global');
  }
  if (Array.from(new Set(linkToIds)).length === 1) {
    throw new Error('Input linkTo cannot remove link to itself');
  }
  if (linkToIds.includes(_constants.InputsModelId.timeline) && linkToIds.includes(_constants.InputsModelId.global)) {
    const socTrends = state.timeline.linkTo.includes(_constants.InputsModelId.socTrends) || state.global.linkTo.includes(_constants.InputsModelId.socTrends) ? [_constants.InputsModelId.socTrends] : [];
    return {
      ...state,
      timeline: {
        ...state.timeline,
        linkTo: []
      },
      global: {
        ...state.global,
        linkTo: socTrends
      },
      // TODO: remove state.socTrends check when socTrendsEnabled feature flag removed
      ...(state.socTrends && socTrends.length ? {
        socTrends: {
          ...state.socTrends,
          linkTo: [_constants.InputsModelId.global]
        }
      } : {})
    };
  }
  if (linkToIds.includes(_constants.InputsModelId.socTrends) && linkToIds.includes(_constants.InputsModelId.global)) {
    const timeline = state.global.linkTo.includes(_constants.InputsModelId.timeline) ? [_constants.InputsModelId.timeline] : [];
    return {
      ...state,
      // TODO: remove state.socTrends check when socTrendsEnabled feature flag removed
      ...(state.socTrends ? {
        socTrends: {
          ...state.socTrends,
          linkTo: []
        }
      } : {}),
      global: {
        ...state.global,
        linkTo: timeline
      },
      ...(timeline.length ? {
        timeline: {
          ...state.timeline,
          linkTo: [_constants.InputsModelId.global]
        }
      } : {})
    };
  }
  return state;
};
exports.removeInputLink = removeInputLink;
const deleteOneQuery = ({
  inputId,
  id,
  state
}) => {
  const queryIndex = state[inputId].queries.findIndex(q => q.id === id);
  return {
    ...state,
    [inputId]: {
      ...(0, _fp.get)(inputId, state),
      queries: queryIndex > -1 ? [...state[inputId].queries.slice(0, queryIndex), ...state[inputId].queries.slice(queryIndex + 1)] : [...state[inputId].queries]
    }
  };
};
exports.deleteOneQuery = deleteOneQuery;
const isQueryInput = inputs => {
  if ('queries' in inputs) {
    return true;
  }
  return false;
};
exports.isQueryInput = isQueryInput;