"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.reorderControlsWithDefaultControls = exports.mergeControls = exports.getPanelsInOrderFromControlsInput = exports.getFilterItemObjListFromControlInput = exports.getFilterControlsComparator = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getPanelsInOrderFromControlsInput = controlInput => {
  const panels = controlInput.panels;
  return Object.values(panels).sort((a, b) => a.order - b.order);
};
exports.getPanelsInOrderFromControlsInput = getPanelsInOrderFromControlsInput;
const getFilterItemObjListFromControlInput = controlInput => {
  const panels = getPanelsInOrderFromControlsInput(controlInput);
  return panels.map(panel => {
    const {
      explicitInput: {
        fieldName,
        selectedOptions,
        title,
        existsSelected,
        exclude,
        hideActionBar
      }
    } = panel;
    return {
      fieldName: fieldName,
      selectedOptions: selectedOptions !== null && selectedOptions !== void 0 ? selectedOptions : [],
      title,
      existsSelected: existsSelected !== null && existsSelected !== void 0 ? existsSelected : false,
      exclude: exclude !== null && exclude !== void 0 ? exclude : false,
      hideActionBar: hideActionBar !== null && hideActionBar !== void 0 ? hideActionBar : false
    };
  });
};
exports.getFilterItemObjListFromControlInput = getFilterItemObjListFromControlInput;
/*
 * mergeControls merges controls based on priority with the default controls
 *
 * @return undefined if all provided controls are empty
 * */
const mergeControls = ({
  controlsWithPriority,
  defaultControlsObj
}) => {
  const highestPriorityControlSet = controlsWithPriority.find(control => !(0, _lodash.isEmpty)(control));
  return highestPriorityControlSet === null || highestPriorityControlSet === void 0 ? void 0 : highestPriorityControlSet.map(singleControl => {
    if (singleControl.fieldName in defaultControlsObj) {
      return {
        ...defaultControlsObj[singleControl.fieldName],
        ...singleControl
      };
    }
    return singleControl;
  });
};
exports.mergeControls = mergeControls;
/**
 * reorderControlsWithPersistentControls reorders the controls such that controls which
 * are persistent in default controls should be upserted in given order
 *
 * */
const reorderControlsWithDefaultControls = args => {
  const {
    controls,
    defaultControls
  } = args;
  const controlsObject = controls.reduce((prev, current) => {
    prev[current.fieldName] = current;
    return prev;
  }, {});
  const defaultControlsObj = defaultControls.reduce((prev, current) => {
    prev[current.fieldName] = current;
    return prev;
  }, {});
  const resultDefaultControls = defaultControls.filter(defaultControl => defaultControl.persist).map(defaultControl => {
    var _controlsObject$defau;
    return {
      ...defaultControl,
      ...((_controlsObject$defau = controlsObject[defaultControl.fieldName]) !== null && _controlsObject$defau !== void 0 ? _controlsObject$defau : {})
    };
  });
  const resultNonPersitantControls = controls.filter(
  // filter out persisting controls since we have already taken
  // in account above
  control => {
    var _defaultControlsObj$c;
    return !((_defaultControlsObj$c = defaultControlsObj[control.fieldName]) !== null && _defaultControlsObj$c !== void 0 && _defaultControlsObj$c.persist);
  }).map(control => {
    var _defaultControlsObj$c2;
    return {
      // insert some default properties from default controls
      // irrespective of whether they are persistent or not.
      ...((_defaultControlsObj$c2 = defaultControlsObj[control.fieldName]) !== null && _defaultControlsObj$c2 !== void 0 ? _defaultControlsObj$c2 : {}),
      ...control
    };
  });
  return [...resultDefaultControls, ...resultNonPersitantControls];
};

/*
 * getFilterControlsComparator provides a comparator that can be used with `isEqualWith` to compare
 * 2 instances of FilterItemObj
 *
 * */
exports.reorderControlsWithDefaultControls = reorderControlsWithDefaultControls;
const getFilterControlsComparator = (...fieldsToCompare) => (filterItemObject1, filterItemObject2) => {
  if (filterItemObject1.length !== filterItemObject2.length) return false;
  const filterItemObjectWithSelectedKeys1 = filterItemObject1.map(v => {
    return (0, _lodash.pick)(v, fieldsToCompare);
  });
  const filterItemObjectWithSelectedKeys2 = filterItemObject2.map(v => {
    return (0, _lodash.pick)(v, fieldsToCompare);
  });
  return (0, _lodash.isEqual)(filterItemObjectWithSelectedKeys1, filterItemObjectWithSelectedKeys2);
};
exports.getFilterControlsComparator = getFilterControlsComparator;