"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.saveWithConfirmation = saveWithConfirmation;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _constants = require("../../constants");
var _confirm_modal_promise = require("./confirm_modal_promise");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Attempts to create the current object using the serialized source. If an object already
 * exists, a warning message requests an overwrite confirmation.
 * @param source - serialized version of this object what will be indexed into elasticsearch.
 * @param savedObject - a simple object that contains properties title and displayName, and getEsType method
 * @param options - options to pass to the saved object create method
 * @param services - provides Kibana services savedObjectsClient and overlays
 * @returns {Promise} - A promise that is resolved with the objects id if the object is
 * successfully indexed. If the overwrite confirmation was rejected, an error is thrown with
 * a confirmRejected = true parameter so that case can be handled differently than
 * a create or index error.
 * @resolved {SavedObject}
 */
async function saveWithConfirmation(source, savedObject, options, services) {
  const {
    savedObjectsClient,
    overlays
  } = services;
  try {
    return await savedObjectsClient.create(savedObject.getEsType(), source, options);
  } catch (err) {
    // record exists, confirm overwriting
    if ((0, _lodash.get)(err, 'res.status') === 409) {
      const confirmMessage = _i18n.i18n.translate('savedObjects.confirmModal.overwriteConfirmationMessage', {
        defaultMessage: 'Are you sure you want to overwrite {title}?',
        values: {
          title: savedObject.title
        }
      });
      const title = _i18n.i18n.translate('savedObjects.confirmModal.overwriteTitle', {
        defaultMessage: 'Overwrite {name}?',
        values: {
          name: savedObject.displayName
        }
      });
      const confirmButtonText = _i18n.i18n.translate('savedObjects.confirmModal.overwriteButtonLabel', {
        defaultMessage: 'Overwrite'
      });
      return (0, _confirm_modal_promise.confirmModalPromise)(confirmMessage, title, confirmButtonText, overlays).then(() => savedObjectsClient.create(savedObject.getEsType(), source, {
        overwrite: true,
        ...options
      })).catch(() => Promise.reject(new Error(_constants.OVERWRITE_REJECTED)));
    }
    return await Promise.reject(err);
  }
}