"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPersistenceRuleTypeWrapper = exports.ALERT_GROUP_INDEX = void 0;
var _datemath = _interopRequireDefault(require("@elastic/datemath"));
var _lodash = require("lodash");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _fp = require("lodash/fp");
var _get_common_alert_fields = require("./get_common_alert_fields");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ALERT_GROUP_INDEX = `${_ruleDataUtils.ALERT_NAMESPACE}.group.index`;
exports.ALERT_GROUP_INDEX = ALERT_GROUP_INDEX;
const augmentAlerts = ({
  alerts,
  options,
  kibanaVersion,
  currentTimeOverride
}) => {
  const commonRuleFields = (0, _get_common_alert_fields.getCommonAlertFields)(options);
  return alerts.map(alert => {
    var _options$maintenanceW;
    return {
      ...alert,
      _source: {
        [_ruleDataUtils.ALERT_START]: currentTimeOverride !== null && currentTimeOverride !== void 0 ? currentTimeOverride : new Date(),
        [_ruleDataUtils.ALERT_LAST_DETECTED]: currentTimeOverride !== null && currentTimeOverride !== void 0 ? currentTimeOverride : new Date(),
        [_ruleDataUtils.VERSION]: kibanaVersion,
        ...(options !== null && options !== void 0 && (_options$maintenanceW = options.maintenanceWindowIds) !== null && _options$maintenanceW !== void 0 && _options$maintenanceW.length ? {
          [_ruleDataUtils.ALERT_MAINTENANCE_WINDOW_IDS]: options.maintenanceWindowIds
        } : {}),
        ...commonRuleFields,
        ...alert._source
      }
    };
  });
};
const mapAlertsToBulkCreate = alerts => {
  return alerts.flatMap(alert => [{
    create: {
      _id: alert._id
    }
  }, alert._source]);
};
const createPersistenceRuleTypeWrapper = ({
  logger,
  ruleDataClient,
  formatAlert
}) => type => {
  return {
    ...type,
    executor: async options => {
      const result = await type.executor({
        ...options,
        services: {
          ...options.services,
          alertWithPersistence: async (alerts, refresh, maxAlerts = undefined, enrichAlerts) => {
            const numAlerts = alerts.length;
            logger.debug(`Found ${numAlerts} alerts.`);
            const ruleDataClientWriter = await ruleDataClient.getWriter({
              namespace: options.spaceId
            });

            // Only write alerts if:
            // - writing is enabled
            //   AND
            //   - rule execution has not been cancelled due to timeout
            //     OR
            //   - if execution has been cancelled due to timeout, if feature flags are configured to write alerts anyway
            const writeAlerts = ruleDataClient.isWriteEnabled() && options.services.shouldWriteAlerts();
            if (writeAlerts && numAlerts) {
              const CHUNK_SIZE = 10000;
              const alertChunks = (0, _lodash.chunk)(alerts, CHUNK_SIZE);
              const filteredAlerts = [];
              for (const alertChunk of alertChunks) {
                const request = {
                  body: {
                    query: {
                      ids: {
                        values: alertChunk.map(alert => alert._id)
                      }
                    },
                    aggs: {
                      uuids: {
                        terms: {
                          field: _ruleDataUtils.ALERT_UUID,
                          size: CHUNK_SIZE
                        }
                      }
                    },
                    size: 0
                  }
                };
                const response = await ruleDataClient.getReader({
                  namespace: options.spaceId
                }).search(request);
                const uuidsMap = {};
                const aggs = response.aggregations;
                if (aggs != null) {
                  aggs.uuids.buckets.forEach(bucket => uuidsMap[bucket.key] = true);
                  const newAlerts = alertChunk.filter(alert => !uuidsMap[alert._id]);
                  filteredAlerts.push(...newAlerts);
                } else {
                  filteredAlerts.push(...alertChunk);
                }
              }
              if (filteredAlerts.length === 0) {
                return {
                  createdAlerts: [],
                  errors: {},
                  alertsWereTruncated: false
                };
              } else if (maxAlerts === 0) {
                return {
                  createdAlerts: [],
                  errors: {},
                  alertsWereTruncated: true
                };
              }
              let enrichedAlerts = filteredAlerts;
              if (enrichAlerts) {
                try {
                  enrichedAlerts = await enrichAlerts(filteredAlerts, {
                    spaceId: options.spaceId
                  });
                } catch (e) {
                  logger.debug('Enrichments failed');
                }
              }
              let alertsWereTruncated = false;
              if (maxAlerts && enrichedAlerts.length > maxAlerts) {
                enrichedAlerts.length = maxAlerts;
                alertsWereTruncated = true;
              }
              const augmentedAlerts = augmentAlerts({
                alerts: enrichedAlerts,
                options,
                kibanaVersion: ruleDataClient.kibanaVersion,
                currentTimeOverride: undefined
              });
              const response = await ruleDataClientWriter.bulk({
                body: mapAlertsToBulkCreate(augmentedAlerts),
                refresh
              });
              if (response == null) {
                return {
                  createdAlerts: [],
                  errors: {},
                  alertsWereTruncated
                };
              }
              const createdAlerts = augmentedAlerts.map((alert, idx) => {
                var _responseItem$_id, _responseItem$_index;
                const responseItem = response.body.items[idx].create;
                return {
                  _id: (_responseItem$_id = responseItem === null || responseItem === void 0 ? void 0 : responseItem._id) !== null && _responseItem$_id !== void 0 ? _responseItem$_id : '',
                  _index: (_responseItem$_index = responseItem === null || responseItem === void 0 ? void 0 : responseItem._index) !== null && _responseItem$_index !== void 0 ? _responseItem$_index : '',
                  ...alert._source
                };
              }).filter((_, idx) => {
                var _response$body$items$;
                return ((_response$body$items$ = response.body.items[idx].create) === null || _response$body$items$ === void 0 ? void 0 : _response$body$items$.status) === 201;
              })
              // Security solution's EQL rule consists of building block alerts which should be filtered out.
              // Building block alerts have additional "kibana.alert.group.index" attribute which is absent for the root alert.
              .filter(alert => !Object.keys(alert).includes(ALERT_GROUP_INDEX));
              createdAlerts.forEach(alert => {
                var _type$getViewInAppRel, _formatAlert;
                return options.services.alertFactory.create(alert._id).replaceState({
                  signals_count: 1
                }).scheduleActions(type.defaultActionGroupId, {
                  rule: (0, _fp.mapKeys)(_fp.snakeCase, {
                    ...options.params,
                    name: options.rule.name,
                    id: options.rule.id
                  }),
                  results_link: (_type$getViewInAppRel = type.getViewInAppRelativeUrl) === null || _type$getViewInAppRel === void 0 ? void 0 : _type$getViewInAppRel.call(type, {
                    rule: {
                      ...options.rule,
                      params: options.params
                    },
                    start: Date.parse(alert[_ruleDataUtils.TIMESTAMP]),
                    end: Date.parse(alert[_ruleDataUtils.TIMESTAMP])
                  }),
                  alerts: [(_formatAlert = formatAlert === null || formatAlert === void 0 ? void 0 : formatAlert(alert)) !== null && _formatAlert !== void 0 ? _formatAlert : alert]
                });
              });
              return {
                createdAlerts,
                errors: (0, _utils.errorAggregator)(response.body, [409]),
                alertsWereTruncated
              };
            } else {
              logger.debug('Writing is disabled.');
              return {
                createdAlerts: [],
                errors: {},
                alertsWereTruncated: false
              };
            }
          },
          alertWithSuppression: async (alerts, suppressionWindow, enrichAlerts, currentTimeOverride) => {
            const ruleDataClientWriter = await ruleDataClient.getWriter({
              namespace: options.spaceId
            });

            // Only write alerts if:
            // - writing is enabled
            //   AND
            //   - rule execution has not been cancelled due to timeout
            //     OR
            //   - if execution has been cancelled due to timeout, if feature flags are configured to write alerts anyway
            const writeAlerts = ruleDataClient.isWriteEnabled() && options.services.shouldWriteAlerts();
            if (writeAlerts && alerts.length > 0) {
              const suppressionWindowStart = _datemath.default.parse(suppressionWindow, {
                forceNow: currentTimeOverride
              });
              if (!suppressionWindowStart) {
                throw new Error('Failed to parse suppression window');
              }
              const suppressionAlertSearchRequest = {
                body: {
                  size: alerts.length,
                  query: {
                    bool: {
                      filter: [{
                        range: {
                          [_ruleDataUtils.ALERT_START]: {
                            gte: suppressionWindowStart.toISOString()
                          }
                        }
                      }, {
                        terms: {
                          [_ruleDataUtils.ALERT_INSTANCE_ID]: alerts.map(alert => alert._source['kibana.alert.instance.id'])
                        }
                      }, {
                        bool: {
                          must_not: {
                            term: {
                              [_ruleDataUtils.ALERT_WORKFLOW_STATUS]: 'closed'
                            }
                          }
                        }
                      }]
                    }
                  },
                  collapse: {
                    field: _ruleDataUtils.ALERT_INSTANCE_ID
                  },
                  sort: [{
                    [_ruleDataUtils.ALERT_START]: {
                      order: 'desc'
                    }
                  }]
                }
              };

              // We use AlertWithSuppressionFields870 explicitly here as the type instead of
              // AlertWithSuppressionFieldsLatest since we're reading alerts rather than writing,
              // so future versions of Kibana may read 8.7.0 version alerts and need to update them
              const response = await ruleDataClient.getReader({
                namespace: options.spaceId
              }).search(suppressionAlertSearchRequest);
              const existingAlertsByInstanceId = response.hits.hits.reduce((acc, hit) => {
                acc[hit._source['kibana.alert.instance.id']] = hit;
                return acc;
              }, {});
              const [duplicateAlerts, newAlerts] = (0, _lodash.partition)(alerts, alert => existingAlertsByInstanceId[alert._source['kibana.alert.instance.id']] != null);
              const duplicateAlertUpdates = duplicateAlerts.flatMap(alert => {
                var _existingAlert$_sourc, _existingAlert$_sourc2;
                const existingAlert = existingAlertsByInstanceId[alert._source['kibana.alert.instance.id']];
                const existingDocsCount = (_existingAlert$_sourc = (_existingAlert$_sourc2 = existingAlert._source) === null || _existingAlert$_sourc2 === void 0 ? void 0 : _existingAlert$_sourc2[_ruleDataUtils.ALERT_SUPPRESSION_DOCS_COUNT]) !== null && _existingAlert$_sourc !== void 0 ? _existingAlert$_sourc : 0;
                return [{
                  update: {
                    _id: existingAlert._id,
                    _index: existingAlert._index,
                    require_alias: false
                  }
                }, {
                  doc: {
                    [_ruleDataUtils.ALERT_LAST_DETECTED]: currentTimeOverride !== null && currentTimeOverride !== void 0 ? currentTimeOverride : new Date(),
                    [_ruleDataUtils.ALERT_SUPPRESSION_END]: alert._source[_ruleDataUtils.ALERT_SUPPRESSION_END],
                    [_ruleDataUtils.ALERT_SUPPRESSION_DOCS_COUNT]: existingDocsCount + alert._source[_ruleDataUtils.ALERT_SUPPRESSION_DOCS_COUNT] + 1
                  }
                }];
              });
              let enrichedAlerts = newAlerts;
              if (enrichAlerts) {
                try {
                  enrichedAlerts = await enrichAlerts(enrichedAlerts, {
                    spaceId: options.spaceId
                  });
                } catch (e) {
                  logger.debug('Enrichments failed');
                }
              }
              const augmentedAlerts = augmentAlerts({
                alerts: enrichedAlerts,
                options,
                kibanaVersion: ruleDataClient.kibanaVersion,
                currentTimeOverride
              });
              const bulkResponse = await ruleDataClientWriter.bulk({
                body: [...duplicateAlertUpdates, ...mapAlertsToBulkCreate(augmentedAlerts)],
                refresh: true
              });
              if (bulkResponse == null) {
                return {
                  createdAlerts: [],
                  errors: {}
                };
              }
              const createdAlerts = augmentedAlerts.map((alert, idx) => {
                var _responseItem$_id2, _responseItem$_index2;
                const responseItem = bulkResponse.body.items[idx + duplicateAlerts.length].create;
                return {
                  _id: (_responseItem$_id2 = responseItem === null || responseItem === void 0 ? void 0 : responseItem._id) !== null && _responseItem$_id2 !== void 0 ? _responseItem$_id2 : '',
                  _index: (_responseItem$_index2 = responseItem === null || responseItem === void 0 ? void 0 : responseItem._index) !== null && _responseItem$_index2 !== void 0 ? _responseItem$_index2 : '',
                  ...alert._source
                };
              }).filter((_, idx) => {
                var _bulkResponse$body$it;
                return ((_bulkResponse$body$it = bulkResponse.body.items[idx + duplicateAlerts.length].create) === null || _bulkResponse$body$it === void 0 ? void 0 : _bulkResponse$body$it.status) === 201;
              })
              // Security solution's EQL rule consists of building block alerts which should be filtered out.
              // Building block alerts have additional "kibana.alert.group.index" attribute which is absent for the root alert.
              .filter(alert => !Object.keys(alert).includes(ALERT_GROUP_INDEX));
              createdAlerts.forEach(alert => {
                var _type$getViewInAppRel2, _formatAlert2;
                return options.services.alertFactory.create(alert._id).replaceState({
                  signals_count: 1
                }).scheduleActions(type.defaultActionGroupId, {
                  rule: (0, _fp.mapKeys)(_fp.snakeCase, {
                    ...options.params,
                    name: options.rule.name,
                    id: options.rule.id
                  }),
                  results_link: (_type$getViewInAppRel2 = type.getViewInAppRelativeUrl) === null || _type$getViewInAppRel2 === void 0 ? void 0 : _type$getViewInAppRel2.call(type, {
                    rule: {
                      ...options.rule,
                      params: options.params
                    },
                    start: Date.parse(alert[_ruleDataUtils.TIMESTAMP]),
                    end: Date.parse(alert[_ruleDataUtils.TIMESTAMP])
                  }),
                  alerts: [(_formatAlert2 = formatAlert === null || formatAlert === void 0 ? void 0 : formatAlert(alert)) !== null && _formatAlert2 !== void 0 ? _formatAlert2 : alert]
                });
              });
              return {
                createdAlerts,
                errors: (0, _utils.errorAggregator)(bulkResponse.body, [409])
              };
            } else {
              logger.debug('Writing is disabled.');
              return {
                createdAlerts: [],
                errors: {}
              };
            }
          }
        }
      });
      return result;
    }
  };
};
exports.createPersistenceRuleTypeWrapper = createPersistenceRuleTypeWrapper;