"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerGenerateCsvFromSavedObjectImmediate = registerGenerateCsvFromSavedObjectImmediate;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _moment = _interopRequireDefault(require("moment"));
var _configSchema = require("@kbn/config-schema");
var _reportingExportTypesCsvCommon = require("@kbn/reporting-export-types-csv-common");
var _constants = require("../../../../common/constants");
var _lib = require("../../../lib");
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const path = _constants.INTERNAL_ROUTES.DOWNLOAD_CSV;
/*
 * This function registers API Endpoints for immediate Reporting jobs. The API inputs are:
 * - saved object type and ID
 * - time range and time zone
 * - application state:
 *     - filters
 *     - query bar
 *     - local (transient) changes the user made to the saved object
 */
function registerGenerateCsvFromSavedObjectImmediate(reporting, parentLogger) {
  const setupDeps = reporting.getPluginSetupDeps();
  const {
    router
  } = setupDeps;
  const useKibanaAccessControl = reporting.getDeprecatedAllowedRoles() === false; // true if deprecated config is turned off
  const kibanaAccessControlTags = useKibanaAccessControl ? ['access:downloadCsv'] : [];

  // This API calls run the SearchSourceImmediate export type's runTaskFn directly
  router.post({
    path,
    validate: {
      body: _configSchema.schema.object({
        columns: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        searchSource: _configSchema.schema.object({}, {
          unknowns: 'allow'
        }),
        browserTimezone: _configSchema.schema.string({
          defaultValue: 'UTC',
          validate: value => _moment.default.tz.zone(value) ? undefined : `Invalid timezone "${typeof value}".`
        }),
        title: _configSchema.schema.string(),
        version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    },
    options: {
      tags: kibanaAccessControlTags,
      access: 'internal'
    }
  }, (0, _common.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
    const counters = (0, _common.getCounters)(req.route.method, path, reporting.getUsageCounter());
    const logger = parentLogger.get(_reportingExportTypesCsvCommon.CSV_SEARCHSOURCE_IMMEDIATE_TYPE);
    const csvSearchSourceImmediateExport = await reporting.getCsvSearchSourceImmediate();
    const stream = new _lib.PassThroughStream();
    const eventLog = reporting.getEventLogger({
      jobtype: _reportingExportTypesCsvCommon.CSV_SEARCHSOURCE_IMMEDIATE_TYPE,
      created_by: user && user.username,
      payload: {
        browserTimezone: req.body.browserTimezone
      }
    });
    const logError = error => {
      logger.error(error);
      eventLog.logError(error);
    };
    try {
      eventLog.logExecutionStart();
      const taskPromise = csvSearchSourceImmediateExport.runTask(null, req.body, context, stream, req).then(output => {
        var _output$metrics;
        logger.info(`Job output size: ${stream.bytesWritten} bytes.`);
        if (!stream.bytesWritten) {
          logger.warn('CSV Job Execution created empty content result');
        }
        eventLog.logExecutionComplete({
          ...((_output$metrics = output.metrics) !== null && _output$metrics !== void 0 ? _output$metrics : {}),
          byteSize: stream.bytesWritten
        });
      }).finally(() => stream.end());
      await Promise.race([stream.firstBytePromise, taskPromise]);
      taskPromise.catch(logError);
      counters.usageCounter();
      return res.ok({
        body: stream,
        headers: {
          'content-type': 'text/csv;charset=utf-8',
          'accept-ranges': 'none'
        }
      });
    } catch (error) {
      logError(error);
      if (error instanceof _boom.default.Boom) {
        const statusCode = error.output.statusCode;
        counters.errorCounter(undefined, statusCode);
        return res.customError({
          statusCode,
          body: error.output.payload.message
        });
      }
      counters.errorCounter(undefined, 500);
      return res.customError({
        statusCode: 500
      });
    }
  }));
}