"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Job = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _moment = _interopRequireDefault(require("moment"));
var _react = _interopRequireDefault(require("react"));
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _reportingCommon = require("@kbn/reporting-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const {
  COMPLETED,
  FAILED,
  PENDING,
  PROCESSING,
  WARNINGS
} = _reportingCommon.JOB_STATUS;
/*
 * This class represents a report job for the UI
 * It can be instantiated with ReportApiJSON: the response data format for the report job APIs
 */
class Job {
  // FIXME: can not use ReportSource['status'] due to type mismatch

  constructor(report) {
    var _report$output, _report$output2, _report$output3, _report$output4, _report$output5, _report$output6;
    (0, _defineProperty2.default)(this, "payload", void 0);
    (0, _defineProperty2.default)(this, "id", void 0);
    (0, _defineProperty2.default)(this, "index", void 0);
    (0, _defineProperty2.default)(this, "objectType", void 0);
    (0, _defineProperty2.default)(this, "title", void 0);
    (0, _defineProperty2.default)(this, "isDeprecated", void 0);
    (0, _defineProperty2.default)(this, "spaceId", void 0);
    (0, _defineProperty2.default)(this, "browserTimezone", void 0);
    (0, _defineProperty2.default)(this, "layout", void 0);
    (0, _defineProperty2.default)(this, "version", void 0);
    (0, _defineProperty2.default)(this, "jobtype", void 0);
    (0, _defineProperty2.default)(this, "created_by", void 0);
    (0, _defineProperty2.default)(this, "created_at", void 0);
    (0, _defineProperty2.default)(this, "started_at", void 0);
    (0, _defineProperty2.default)(this, "completed_at", void 0);
    (0, _defineProperty2.default)(this, "status", void 0);
    (0, _defineProperty2.default)(this, "attempts", void 0);
    (0, _defineProperty2.default)(this, "max_attempts", void 0);
    (0, _defineProperty2.default)(this, "timeout", void 0);
    (0, _defineProperty2.default)(this, "kibana_name", void 0);
    (0, _defineProperty2.default)(this, "kibana_id", void 0);
    (0, _defineProperty2.default)(this, "size", void 0);
    (0, _defineProperty2.default)(this, "content_type", void 0);
    (0, _defineProperty2.default)(this, "csv_contains_formulas", void 0);
    (0, _defineProperty2.default)(this, "max_size_reached", void 0);
    (0, _defineProperty2.default)(this, "metrics", void 0);
    (0, _defineProperty2.default)(this, "warnings", void 0);
    (0, _defineProperty2.default)(this, "error_code", void 0);
    (0, _defineProperty2.default)(this, "locatorParams", void 0);
    (0, _defineProperty2.default)(this, "queue_time_ms", void 0);
    (0, _defineProperty2.default)(this, "execution_time_ms", void 0);
    this.id = report.id;
    this.index = report.index;
    this.payload = report.payload;
    this.jobtype = report.jobtype;
    this.objectType = report.payload.objectType;
    this.title = report.payload.title;
    this.layout = report.payload.layout;
    this.version = report.payload.version;
    this.created_by = report.created_by;
    this.created_at = report.created_at;
    this.started_at = report.started_at;
    this.completed_at = report.completed_at;
    this.status = report.status;
    this.attempts = report.attempts;
    this.max_attempts = report.max_attempts;
    this.timeout = report.timeout;
    this.kibana_name = report.kibana_name;
    this.kibana_id = report.kibana_id;
    this.browserTimezone = report.payload.browserTimezone;
    this.size = (_report$output = report.output) === null || _report$output === void 0 ? void 0 : _report$output.size;
    this.content_type = (_report$output2 = report.output) === null || _report$output2 === void 0 ? void 0 : _report$output2.content_type;
    this.isDeprecated = report.payload.isDeprecated || false;
    this.spaceId = report.payload.spaceId;
    this.csv_contains_formulas = (_report$output3 = report.output) === null || _report$output3 === void 0 ? void 0 : _report$output3.csv_contains_formulas;
    this.max_size_reached = (_report$output4 = report.output) === null || _report$output4 === void 0 ? void 0 : _report$output4.max_size_reached;
    this.warnings = (_report$output5 = report.output) === null || _report$output5 === void 0 ? void 0 : _report$output5.warnings;
    this.error_code = (_report$output6 = report.output) === null || _report$output6 === void 0 ? void 0 : _report$output6.error_code;
    this.locatorParams = report.payload.locatorParams;
    this.metrics = report.metrics;
    this.queue_time_ms = report.queue_time_ms;
    this.execution_time_ms = report.execution_time_ms;
  }
  isSearch() {
    return this.objectType === 'search';
  }
  getStatusMessage() {
    const status = this.status;
    let smallMessage;
    if (status === PENDING) {
      smallMessage = _i18n.i18n.translate('xpack.reporting.jobStatusDetail.pendingStatusReachedText', {
        defaultMessage: 'Waiting for job to process.'
      });
    } else if (status === PROCESSING) {
      smallMessage = _i18n.i18n.translate('xpack.reporting.jobStatusDetail.attemptXofY', {
        defaultMessage: 'Attempt {attempts} of {max_attempts}.',
        values: {
          attempts: this.attempts,
          max_attempts: this.max_attempts
        }
      });
    } else if (this.getWarnings()) {
      smallMessage = _i18n.i18n.translate('xpack.reporting.jobStatusDetail.warningsText', {
        defaultMessage: 'See report info for warnings.'
      });
    } else if (this.getError()) {
      smallMessage = _i18n.i18n.translate('xpack.reporting.jobStatusDetail.errorText', {
        defaultMessage: 'See report info for error details.'
      });
    }
    let deprecatedMessage;
    if (this.isDeprecated) {
      deprecatedMessage = /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
        size: "s"
      }, ' ', /*#__PURE__*/_react.default.createElement(_eui.EuiTextColor, {
        color: "warning"
      }, _i18n.i18n.translate('xpack.reporting.jobStatusDetail.deprecatedText', {
        defaultMessage: `This is a deprecated export type. Automation of this report will need to be re-created for compatibility with future versions of Kibana.`
      })));
    }
    if (smallMessage) {
      return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
        size: "s"
      }, /*#__PURE__*/_react.default.createElement(_eui.EuiTextColor, {
        color: "subdued"
      }, smallMessage)), deprecatedMessage ? deprecatedMessage : null);
    }
    return null;
  }
  get prettyStatus() {
    var _jobStatusLabelsMap$g;
    return (_jobStatusLabelsMap$g = jobStatusLabelsMap.get(this.status)) !== null && _jobStatusLabelsMap$g !== void 0 ? _jobStatusLabelsMap$g : _i18n.i18n.translate('xpack.reporting.jobStatusDetail.unknownText', {
      defaultMessage: 'Unknown'
    });
  }
  get canLinkToKibanaApp() {
    return Boolean(this.locatorParams);
  }
  get isDownloadReady() {
    return this.status === _reportingCommon.JOB_STATUS.COMPLETED || this.status === _reportingCommon.JOB_STATUS.WARNINGS;
  }
  get prettyJobTypeName() {
    switch (this.jobtype) {
      case 'printable_pdf':
      case 'printable_pdf_v2':
        return _i18n.i18n.translate('xpack.reporting.jobType.pdfOutputName', {
          defaultMessage: 'PDF'
        });
      case 'PNG':
      case 'PNGV2':
        return _i18n.i18n.translate('xpack.reporting.jobType.pngOutputName', {
          defaultMessage: 'PNG'
        });
      case 'csv_v2':
      case 'csv_searchsource':
        return _i18n.i18n.translate('xpack.reporting.jobType.csvOutputName', {
          defaultMessage: 'CSV'
        });
      default:
        return undefined;
    }
  }
  get prettyTimeout() {
    if (this.timeout == null) {
      return _i18n.i18n.translate('xpack.reporting.jobStatusDetail.timeoutSecondsUnknown', {
        defaultMessage: 'Unknown'
      });
    }
    const seconds = this.timeout / 1000;
    return _i18n.i18n.translate('xpack.reporting.jobStatusDetail.timeoutSeconds', {
      defaultMessage: '{timeout} seconds',
      values: {
        timeout: seconds
      }
    });
  }

  /**
   * Returns a user friendly version of the report job creation date
   */
  getCreatedAtDate() {
    return this.formatDate(this.created_at);
  }

  /**
   * Returns a user friendly version of the user that created the report job
   */
  getCreatedBy() {
    return this.created_by || _i18n.i18n.translate('xpack.reporting.jobCreatedBy.unknownUserPlaceholderText', {
      defaultMessage: 'Unknown'
    });
  }
  getCreatedAtLabel() {
    if (this.created_by) {
      return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement("div", null, this.formatDate(this.created_at)), /*#__PURE__*/_react.default.createElement("span", null, this.created_by));
    }
    return this.formatDate(this.created_at);
  }

  /*
   * We use `output.warnings` to show the error of a failed report job,
   * and to show warnings of a job that completed with warnings.
   */

  // There is no error unless the status is 'failed'
  getError() {
    if (this.status === FAILED) {
      return this.warnings;
    }
  }
  getDeprecatedMessage() {
    if (this.isDeprecated) {
      return _i18n.i18n.translate('xpack.reporting.jobWarning.exportTypeDeprecated', {
        defaultMessage: 'This is a deprecated export type. Automation of this report will need to be re-created for compatibility with future versions of Kibana.'
      });
    }
  }
  getWarnings() {
    var _this$warnings;
    const warnings = [];
    const deprecatedMessage = this.getDeprecatedMessage();
    if (deprecatedMessage) {
      warnings.push(deprecatedMessage);
    }
    if (this.csv_contains_formulas) {
      warnings.push(_i18n.i18n.translate('xpack.reporting.jobWarning.csvContainsFormulas', {
        defaultMessage: 'Your CSV contains characters that spreadsheet applications might interpret as formulas.'
      }));
    }
    if (this.max_size_reached) {
      warnings.push(_i18n.i18n.translate('xpack.reporting.jobWarning.maxSizeReachedTooltip', {
        defaultMessage: 'Your search reached the max size and contains partial data.'
      }));
    }

    // warnings could contain the failure message
    if (this.status !== FAILED && (_this$warnings = this.warnings) !== null && _this$warnings !== void 0 && _this$warnings.length) {
      warnings.push(...this.warnings);
    }
    if (warnings.length) {
      return /*#__PURE__*/_react.default.createElement("ul", null, warnings.map((w, i) => {
        return /*#__PURE__*/_react.default.createElement("li", {
          key: `warning-key-${i}`
        }, w);
      }));
    }
  }
  getPrettyStatusTimestamp() {
    return this.formatDate(this.getStatusTimestamp());
  }
  formatDate(timestamp) {
    try {
      return (0, _moment.default)(timestamp).format('YYYY-MM-DD @ hh:mm A');
    } catch (error) {
      // ignore parse error and display unformatted value
      return timestamp;
    }
  }
  getStatusTimestamp() {
    const status = this.status;
    if (status === PROCESSING && this.started_at) {
      return this.started_at;
    }
    if (this.completed_at && [COMPLETED, FAILED, WARNINGS].includes(status)) {
      return this.completed_at;
    }
    return this.created_at;
  }
}
exports.Job = Job;
const jobStatusLabelsMap = new Map([[PENDING, _i18n.i18n.translate('xpack.reporting.jobStatuses.pendingText', {
  defaultMessage: 'Pending'
})], [PROCESSING, _i18n.i18n.translate('xpack.reporting.jobStatuses.processingText', {
  defaultMessage: 'Processing'
})], [COMPLETED, _i18n.i18n.translate('xpack.reporting.jobStatuses.completedText', {
  defaultMessage: 'Completed' // NOTE: a job is `completed` not `completed_with_warings` if it has reached max size or possibly contains csv characters
})], [WARNINGS, _i18n.i18n.translate('xpack.reporting.jobStatuses.warningText', {
  defaultMessage: 'Completed'
})], [FAILED, _i18n.i18n.translate('xpack.reporting.jobStatuses.failedText', {
  defaultMessage: 'Failed'
})]]);