"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createCollectorPackagePolicy = createCollectorPackagePolicy;
exports.createSymbolizerPackagePolicy = createSymbolizerPackagePolicy;
exports.generateSecretToken = generateSecretToken;
exports.getVarsFor = getVarsFor;
exports.removeProfilingFromApmPackagePolicy = removeProfilingFromApmPackagePolicy;
var _registry = require("@kbn/fleet-plugin/server/services/epm/registry");
var _common = require("@kbn/profiling-data-access-plugin/common");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CLOUD_AGENT_POLICY_ID = 'policy-elastic-agent-on-cloud';
function generateSecretToken() {
  let result = '';
  const characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
  for (let i = 0; i < 16; i++) {
    const randomIndex = Math.floor(Math.random() * characters.length);
    result += characters.charAt(randomIndex);
  }
  return result;
}
function getVarsFor({
  config,
  includeSecretToken
}) {
  const configKeys = Object.keys(config);
  if (includeSecretToken) {
    configKeys.push('secret_token');
  }
  return configKeys.reduce((acc, currKey) => {
    const value = currKey === 'secret_token' ? generateSecretToken() : config[currKey];
    const type = typeof value === 'boolean' ? 'bool' : 'text';
    return {
      ...acc,
      [currKey]: {
        type,
        value
      }
    };
  }, {});
}
async function createCollectorPackagePolicy({
  client,
  soClient,
  packagePolicyClient,
  config
}) {
  const packageName = 'profiler_collector';
  const {
    version
  } = await (0, _registry.fetchFindLatestPackageOrThrow)(packageName, {
    prerelease: true
  });
  const packagePolicy = {
    policy_id: CLOUD_AGENT_POLICY_ID,
    enabled: true,
    package: {
      name: packageName,
      title: 'Universal Profiling Collector',
      version
    },
    name: _common.COLLECTOR_PACKAGE_POLICY_NAME,
    namespace: 'default',
    inputs: [{
      policy_template: 'universal_profiling_collector',
      enabled: true,
      streams: [],
      type: 'pf-elastic-collector',
      vars: config !== null && config !== void 0 && config.collector ? getVarsFor({
        config: config.collector,
        includeSecretToken: true
      }) : {}
    }]
  };
  const esClient = client.getEsClient();
  await packagePolicyClient.create(soClient, esClient, packagePolicy, {
    force: true
  });
}
async function createSymbolizerPackagePolicy({
  client,
  soClient,
  packagePolicyClient,
  config
}) {
  const packageName = 'profiler_symbolizer';
  const {
    version
  } = await (0, _registry.fetchFindLatestPackageOrThrow)(packageName, {
    prerelease: true
  });
  const packagePolicy = {
    policy_id: CLOUD_AGENT_POLICY_ID,
    enabled: true,
    package: {
      name: packageName,
      title: 'Universal Profiling Symbolizer',
      version
    },
    name: _common.SYMBOLIZER_PACKAGE_POLICY_NAME,
    namespace: 'default',
    inputs: [{
      policy_template: 'universal_profiling_symbolizer',
      enabled: true,
      streams: [],
      type: 'pf-elastic-symbolizer',
      // doesnt have secret token
      vars: config !== null && config !== void 0 && config.symbolizer ? getVarsFor({
        config: config.symbolizer,
        includeSecretToken: false
      }) : {}
    }]
  };
  const esClient = client.getEsClient();
  await packagePolicyClient.create(soClient, esClient, packagePolicy, {
    force: true
  });
}
async function removeProfilingFromApmPackagePolicy({
  client,
  soClient,
  packagePolicyClient
}) {
  const apmPackagePolicy = await (0, _common.getApmPolicy)({
    packagePolicyClient,
    soClient
  });
  if (!apmPackagePolicy) {
    throw new Error(`Could not find APM package policy`);
  }
  const esClient = client.getEsClient();
  // remove profiling from apm-server config
  const newPackagePolicy = (0, _lodash.omit)(apmPackagePolicy, "inputs[0].config['apm-server'].value.profiling");
  await packagePolicyClient.update(soClient, esClient, _common.ELASTIC_CLOUD_APM_POLICY, newPackagePolicy);
}