"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useEditableSettings = useEditableSettings;
var _public = require("@kbn/kibana-react-plugin/public");
var _react = require("react");
var _public2 = require("@kbn/advanced-settings-plugin/public");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getEditableConfig({
  settingsKeys,
  uiSettings
}) {
  if (!uiSettings) {
    return {};
  }
  const uiSettingsDefinition = uiSettings.getAll();
  const config = {};
  settingsKeys.forEach(key => {
    const settingDef = uiSettingsDefinition === null || uiSettingsDefinition === void 0 ? void 0 : uiSettingsDefinition[key];
    if (settingDef) {
      const editableConfig = (0, _public2.toEditableConfig)({
        def: settingDef,
        name: key,
        value: settingDef.userValue,
        isCustom: uiSettings.isCustom(key),
        isOverridden: uiSettings.isOverridden(key)
      });
      config[key] = editableConfig;
    }
  });
  return config;
}
function useEditableSettings(app, settingsKeys) {
  const {
    services: {
      settings
    }
  } = (0, _public.useKibana)();
  const [isSaving, setIsSaving] = (0, _react.useState)(false);
  const [forceReloadSettings, setForceReloadSettings] = (0, _react.useState)(0);
  const [unsavedChanges, setUnsavedChanges] = (0, _react.useState)({});
  const settingsEditableConfig = (0, _react.useMemo)(() => {
    return getEditableConfig({
      settingsKeys,
      uiSettings: settings === null || settings === void 0 ? void 0 : settings.client
    });
  },
  // eslint-disable-next-line react-hooks/exhaustive-deps
  [settings, settingsKeys, forceReloadSettings]);
  function handleFieldChange(key, fieldState) {
    setUnsavedChanges(state => {
      const newState = {
        ...state
      };
      const {
        value,
        defVal
      } = settingsEditableConfig[key];
      const currentValue = value === undefined ? defVal : value;
      if (currentValue === fieldState.value) {
        // Delete property from unsaved object if user changes it to the value that was already saved
        delete newState[key];
      } else {
        newState[key] = fieldState;
      }
      return newState;
    });
  }
  function cleanUnsavedChanges() {
    setUnsavedChanges({});
  }
  async function saveAll() {
    if (settings && !(0, _lodash.isEmpty)(unsavedChanges)) {
      try {
        setIsSaving(true);
        const arr = Object.entries(unsavedChanges).map(([key, fieldState]) => settings.client.set(key, fieldState.value));
        await Promise.all(arr);
        setForceReloadSettings(state => ++state);
        cleanUnsavedChanges();
      } finally {
        setIsSaving(false);
      }
    }
  }
  return {
    settingsEditableConfig,
    unsavedChanges,
    handleFieldChange,
    saveAll,
    isSaving,
    cleanUnsavedChanges
  };
}