"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultTransformManager = void 0;
var _errors = require("../../errors");
var _retry = require("../../utils/retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DefaultTransformManager {
  constructor(generators, esClient, logger) {
    this.generators = generators;
    this.esClient = esClient;
    this.logger = logger;
  }
  async install(slo) {
    const generator = this.generators[slo.indicator.type];
    if (!generator) {
      this.logger.error(`No transform generator found for indicator type [${slo.indicator.type}]`);
      throw new Error(`Unsupported indicator type [${slo.indicator.type}]`);
    }
    const transformParams = generator.getTransformParams(slo);
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.transform.putTransform(transformParams), {
        logger: this.logger
      });
    } catch (err) {
      var _err$meta, _err$meta$body, _err$meta$body$error;
      this.logger.error(`Cannot create SLO transform for indicator type [${slo.indicator.type}]`);
      if (((_err$meta = err.meta) === null || _err$meta === void 0 ? void 0 : (_err$meta$body = _err$meta.body) === null || _err$meta$body === void 0 ? void 0 : (_err$meta$body$error = _err$meta$body.error) === null || _err$meta$body$error === void 0 ? void 0 : _err$meta$body$error.type) === 'security_exception') {
        throw new _errors.SecurityException(err.meta.body.error.reason);
      }
      throw err;
    }
    return transformParams.transform_id;
  }
  async preview(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.transform.previewTransform({
        transform_id: transformId
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.error(`Cannot preview SLO transform [${transformId}]`);
      throw err;
    }
  }
  async start(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.transform.startTransform({
        transform_id: transformId
      }, {
        ignore: [409]
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.error(`Cannot start SLO transform [${transformId}]`);
      throw err;
    }
  }
  async stop(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.transform.stopTransform({
        transform_id: transformId,
        wait_for_completion: true,
        force: true
      }, {
        ignore: [404]
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.error(`Cannot stop SLO transform [${transformId}]`);
      throw err;
    }
  }
  async uninstall(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.transform.deleteTransform({
        transform_id: transformId,
        force: true
      }, {
        ignore: [404]
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.error(`Cannot delete SLO transform [${transformId}]`);
      throw err;
    }
  }
}
exports.DefaultTransformManager = DefaultTransformManager;