"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HistogramTransformGenerator = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _errors = require("../../../errors");
var _slo_transform_template = require("../../../assets/transform_templates/slo_transform_template");
var _ = require(".");
var _constants = require("../../../../common/slo/constants");
var _aggregations = require("../aggregations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class HistogramTransformGenerator extends _.TransformGenerator {
  getTransformParams(slo) {
    if (!_sloSchema.histogramIndicatorSchema.is(slo.indicator)) {
      throw new _errors.InvalidTransformError(`Cannot handle SLO of indicator type: ${slo.indicator.type}`);
    }
    return (0, _slo_transform_template.getSLOTransformTemplate)(this.buildTransformId(slo), this.buildDescription(slo), this.buildSource(slo, slo.indicator), this.buildDestination(), this.buildCommonGroupBy(slo, slo.indicator.params.timestampField), this.buildAggregations(slo, slo.indicator), this.buildSettings(slo, slo.indicator.params.timestampField));
  }
  buildTransformId(slo) {
    return (0, _constants.getSLOTransformId)(slo.id, slo.revision);
  }
  buildSource(slo, indicator) {
    return {
      index: (0, _.parseIndex)(indicator.params.index),
      runtime_mappings: this.buildCommonRuntimeMappings(slo),
      query: {
        bool: {
          filter: [{
            range: {
              [indicator.params.timestampField]: {
                gte: `now-${slo.timeWindow.duration.format()}/d`
              }
            }
          }, (0, _.getElastichsearchQueryOrThrow)(indicator.params.filter)]
        }
      }
    };
  }
  buildDestination() {
    return {
      pipeline: _constants.SLO_INGEST_PIPELINE_NAME,
      index: _constants.SLO_DESTINATION_INDEX_NAME
    };
  }
  buildAggregations(slo, indicator) {
    const getHistogramIndicatorAggregations = new _aggregations.GetHistogramIndicatorAggregation(indicator);
    return {
      ...getHistogramIndicatorAggregations.execute({
        type: 'good',
        aggregationKey: 'slo.numerator'
      }),
      ...getHistogramIndicatorAggregations.execute({
        type: 'total',
        aggregationKey: 'slo.denominator'
      }),
      ...(_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod) && {
        'slo.isGoodSlice': {
          bucket_script: {
            buckets_path: {
              goodEvents: 'slo.numerator>value',
              totalEvents: 'slo.denominator>value'
            },
            script: `params.goodEvents / params.totalEvents >= ${slo.objective.timesliceTarget} ? 1 : 0`
          }
        }
      })
    };
  }
}
exports.HistogramTransformGenerator = HistogramTransformGenerator;