"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CreateSLO = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _uuid = require("uuid");
var _constants = require("../../../common/slo/constants");
var _slo_summary_pipeline_template = require("../../assets/ingest_templates/slo_summary_pipeline_template");
var _models = require("../../domain/models");
var _services = require("../../domain/services");
var _retry = require("../../utils/retry");
var _create_temp_summary = require("./summary_transform_generator/helpers/create_temp_summary");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CreateSLO {
  constructor(esClient, repository, transformManager, summaryTransformManager, logger, spaceId) {
    this.esClient = esClient;
    this.repository = repository;
    this.transformManager = transformManager;
    this.summaryTransformManager = summaryTransformManager;
    this.logger = logger;
    this.spaceId = spaceId;
  }
  async execute(params) {
    const slo = this.toSLO(params);
    (0, _services.validateSLO)(slo);
    await this.repository.save(slo, {
      throwOnConflict: true
    });
    const rollupTransformId = (0, _constants.getSLOTransformId)(slo.id, slo.revision);
    const summaryTransformId = (0, _constants.getSLOSummaryTransformId)(slo.id, slo.revision);
    try {
      await this.transformManager.install(slo);
      await this.transformManager.start(rollupTransformId);
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.ingest.putPipeline((0, _slo_summary_pipeline_template.getSLOSummaryPipelineTemplate)(slo, this.spaceId)), {
        logger: this.logger
      });
      await this.summaryTransformManager.install(slo);
      await this.summaryTransformManager.start(summaryTransformId);
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.index({
        index: _constants.SLO_SUMMARY_TEMP_INDEX_NAME,
        id: `slo-${slo.id}`,
        document: (0, _create_temp_summary.createTempSummaryDocument)(slo, this.spaceId),
        refresh: true
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.error(`Cannot install the SLO [id: ${slo.id}, revision: ${slo.revision}]. Rolling back.`);
      await this.summaryTransformManager.stop(summaryTransformId);
      await this.summaryTransformManager.uninstall(summaryTransformId);
      await this.transformManager.stop(rollupTransformId);
      await this.transformManager.uninstall(rollupTransformId);
      await this.esClient.ingest.deletePipeline({
        id: (0, _constants.getSLOSummaryPipelineId)(slo.id, slo.revision)
      }, {
        ignore: [404]
      });
      await this.repository.deleteById(slo.id);
      throw err;
    }
    return this.toResponse(slo);
  }
  toSLO(params) {
    var _params$id, _params$settings$sync, _params$settings, _params$settings$freq, _params$settings2, _params$tags;
    const now = new Date();
    return {
      ...params,
      id: (_params$id = params.id) !== null && _params$id !== void 0 ? _params$id : (0, _uuid.v4)(),
      settings: {
        syncDelay: (_params$settings$sync = (_params$settings = params.settings) === null || _params$settings === void 0 ? void 0 : _params$settings.syncDelay) !== null && _params$settings$sync !== void 0 ? _params$settings$sync : new _models.Duration(1, _models.DurationUnit.Minute),
        frequency: (_params$settings$freq = (_params$settings2 = params.settings) === null || _params$settings2 === void 0 ? void 0 : _params$settings2.frequency) !== null && _params$settings$freq !== void 0 ? _params$settings$freq : new _models.Duration(1, _models.DurationUnit.Minute)
      },
      revision: 1,
      enabled: true,
      tags: (_params$tags = params.tags) !== null && _params$tags !== void 0 ? _params$tags : [],
      createdAt: now,
      updatedAt: now,
      groupBy: !!params.groupBy ? params.groupBy : _sloSchema.ALL_VALUE,
      version: _constants.SLO_MODEL_VERSION
    };
  }
  toResponse(slo) {
    return {
      id: slo.id
    };
  }
}
exports.CreateSLO = CreateSLO;