"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useConversation = useConversation;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _react = require("react");
var _i18n2 = require("../i18n");
var _use_abortable_async = require("./use_abortable_async");
var _use_chat = require("./use_chat");
var _use_kibana = require("./use_kibana");
var _use_observability_ai_assistant = require("./use_observability_ai_assistant");
var _use_once = require("./use_once");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createNewConversation({
  title = _i18n2.EMPTY_CONVERSATION_TITLE
} = {}) {
  return {
    '@timestamp': new Date().toISOString(),
    messages: [],
    conversation: {
      title
    },
    labels: {},
    numeric_labels: {},
    public: false
  };
}
const DEFAULT_INITIAL_MESSAGES = [];
function useConversation({
  initialConversationId: initialConversationIdFromProps,
  initialMessages: initialMessagesFromProps = DEFAULT_INITIAL_MESSAGES,
  initialTitle: initialTitleFromProps,
  chatService,
  connectorId,
  onConversationUpdate
}) {
  const service = (0, _use_observability_ai_assistant.useObservabilityAIAssistant)();
  const {
    services: {
      notifications
    }
  } = (0, _use_kibana.useKibana)();
  const initialConversationId = (0, _use_once.useOnce)(initialConversationIdFromProps);
  const initialMessages = (0, _use_once.useOnce)(initialMessagesFromProps);
  const initialTitle = (0, _use_once.useOnce)(initialTitleFromProps);
  if (initialMessages.length && initialConversationId) {
    throw new Error('Cannot set initialMessages if initialConversationId is set');
  }
  const update = nextConversationObject => {
    return service.callApi(`PUT /internal/observability_ai_assistant/conversation/{conversationId}`, {
      signal: null,
      params: {
        path: {
          conversationId: nextConversationObject.conversation.id
        },
        body: {
          conversation: (0, _lodash.merge)({
            '@timestamp': nextConversationObject['@timestamp'],
            conversation: {
              id: nextConversationObject.conversation.id
            }
          }, (0, _lodash.omit)(nextConversationObject, 'conversation.last_updated', 'namespace', 'user'))
        }
      }
    }).catch(err => {
      notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('xpack.observabilityAiAssistant.errorUpdatingConversation', {
          defaultMessage: 'Could not update conversation'
        })
      });
      throw err;
    });
  };
  const {
    next,
    messages,
    setMessages,
    state,
    stop
  } = (0, _use_chat.useChat)({
    initialMessages,
    initialConversationId,
    chatService,
    connectorId,
    onConversationUpdate: event => {
      setDisplayedConversationId(event.conversation.id);
      onConversationUpdate === null || onConversationUpdate === void 0 ? void 0 : onConversationUpdate({
        conversation: event.conversation
      });
    },
    persist: true
  });
  const [displayedConversationId, setDisplayedConversationId] = (0, _react.useState)(initialConversationId);
  const conversation = (0, _use_abortable_async.useAbortableAsync)(({
    signal
  }) => {
    if (!displayedConversationId) {
      const nextConversation = createNewConversation({
        title: initialTitle
      });
      return nextConversation;
    }
    return service.callApi('GET /internal/observability_ai_assistant/conversation/{conversationId}', {
      signal,
      params: {
        path: {
          conversationId: displayedConversationId
        }
      }
    }).then(nextConversation => {
      setMessages(nextConversation.messages);
      return nextConversation;
    }).catch(error => {
      setMessages([]);
      throw error;
    });
  }, [displayedConversationId, initialTitle], {
    defaultValue: () => {
      if (!displayedConversationId) {
        const nextConversation = createNewConversation({
          title: initialTitle
        });
        return nextConversation;
      }
      return undefined;
    }
  });
  return {
    conversation,
    state,
    next,
    stop,
    messages,
    saveTitle: title => {
      if (!displayedConversationId || !conversation.value) {
        throw new Error('Cannot save title if conversation is not stored');
      }
      const nextConversation = (0, _lodash.merge)({}, conversation.value, {
        conversation: {
          title
        }
      });
      return update(nextConversation).then(() => {
        return conversation.refresh();
      }).then(() => {
        onConversationUpdate === null || onConversationUpdate === void 0 ? void 0 : onConversationUpdate(nextConversation);
      });
    }
  };
}