"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.modelManagementRoutes = modelManagementRoutes;
var _configSchema = require("@kbn/config-schema");
var _app = require("../../common/constants/app");
var _error_wrapper = require("../client/error_wrapper");
var _model_management = require("../models/model_management");
var _saved_objects = require("./schemas/saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function modelManagementRoutes({
  router,
  routeGuard,
  getEnabledFeatures
}) {
  /**
   * @apiGroup ModelManagement
   *
   * @api {get} /internal/ml/model_management/nodes_overview Get node overview about the models allocation
   * @apiName GetModelManagementNodesOverview
   * @apiDescription Retrieves the list of ML nodes with memory breakdown and allocated models info
   */
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/model_management/nodes_overview`,
    access: 'internal',
    options: {
      tags: ['access:ml:canViewMlNodes', 'access:ml:canGetDataFrameAnalytics', 'access:ml:canGetJobs', 'access:ml:canGetTrainedModels']
    }
  }).addVersion({
    version: '1',
    validate: {}
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    mlClient,
    response
  }) => {
    try {
      const memoryUsageService = new _model_management.MemoryUsageService(mlClient, getEnabledFeatures());
      const result = await memoryUsageService.getNodesOverview();
      return response.ok({
        body: result
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));

  /**
   * @apiGroup ModelManagement
   *
   * @api {get} /internal/ml/model_management/memory_usage Memory usage for jobs and trained models
   * @apiName GetModelManagementMemoryUsage
   * @apiDescription Returns the memory usage for jobs and trained models
   */
  router.versioned.get({
    path: `${_app.ML_INTERNAL_BASE_PATH}/model_management/memory_usage`,
    access: 'internal',
    options: {
      tags: ['access:ml:canViewMlNodes', 'access:ml:canGetDataFrameAnalytics', 'access:ml:canGetJobs', 'access:ml:canGetTrainedModels']
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        query: _configSchema.schema.object({
          type: _configSchema.schema.maybe(_saved_objects.itemTypeLiterals),
          node: _configSchema.schema.maybe(_configSchema.schema.string()),
          showClosedJobs: _configSchema.schema.maybe(_configSchema.schema.boolean())
        })
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    mlClient,
    response,
    request
  }) => {
    try {
      const memoryUsageService = new _model_management.MemoryUsageService(mlClient, getEnabledFeatures());
      return response.ok({
        body: await memoryUsageService.getMemorySizes(request.query.type, request.query.node, request.query.showClosedJobs)
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
}