"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFunctionDescription = void 0;
exports.isMetricDetector = isMetricDetector;
var _i18n = require("@kbn/i18n");
var _rxjs = require("rxjs");
var _mlAnomalyUtils = require("@kbn/ml-anomaly-utils");
var _results_service = require("../services/results_service");
var _get_controls_for_detector = require("./get_controls_for_detector");
var _get_criteria_fields = require("./get_criteria_fields");
var _get_viewable_detectors = require("./timeseriesexplorer_utils/get_viewable_detectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isMetricDetector(selectedJob, selectedDetectorIndex) {
  const detectors = (0, _get_viewable_detectors.getViewableDetectors)(selectedJob);
  if (Array.isArray(detectors) && detectors.length >= selectedDetectorIndex) {
    const detector = selectedJob.analysis_config.detectors[selectedDetectorIndex];
    if ((detector === null || detector === void 0 ? void 0 : detector.function) === _mlAnomalyUtils.ML_JOB_AGGREGATION.METRIC) {
      return true;
    }
  }
  return false;
}

/**
 * Get the function description from the record with the highest anomaly score
 */
const getFunctionDescription = async ({
  selectedDetectorIndex,
  selectedEntities,
  selectedJobId,
  selectedJob
}, toastNotificationService) => {
  // if the detector's function is metric, fetch the highest scoring anomaly record
  // and set to plot the function_description (avg/min/max) of that record by default
  if (!isMetricDetector(selectedJob, selectedDetectorIndex)) return;
  const entityControls = (0, _get_controls_for_detector.getControlsForDetector)(selectedDetectorIndex, selectedEntities, selectedJobId);
  const criteriaFields = (0, _get_criteria_fields.getCriteriaFields)(selectedDetectorIndex, entityControls);
  try {
    const resp = await (0, _rxjs.lastValueFrom)(_results_service.mlResultsService.getRecordsForCriteria([selectedJob.job_id], criteriaFields, 0, null, null, 1));
    if (Array.isArray(resp === null || resp === void 0 ? void 0 : resp.records) && resp.records.length === 1) {
      // grabbing first record because records should have already been sorted by score desc
      const highestScoringAnomaly = resp.records[0];
      return highestScoringAnomaly === null || highestScoringAnomaly === void 0 ? void 0 : highestScoringAnomaly.function_description;
    }
    // if there's no anomaly found, auto default to plotting the mean
    return _mlAnomalyUtils.ES_AGGREGATION.AVG;
  } catch (error) {
    toastNotificationService.displayErrorToast(error, _i18n.i18n.translate('xpack.ml.timeSeriesExplorer.highestAnomalyScoreErrorToastTitle', {
      defaultMessage: 'An error occurred getting record with the highest anomaly score'
    }));
  }
};
exports.getFunctionDescription = getFunctionDescription;