"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LabelPositionProperty = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _i18n = require("@kbn/i18n");
var _style_property = require("./style_property");
var _constants = require("../../../../../common/constants");
var _vector_style_defaults = require("../vector_style_defaults");
var _get_vector_style_label = require("../components/get_vector_style_label");
var _style_prop_editor = require("../components/style_prop_editor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class LabelPositionProperty extends _style_property.AbstractStyleProperty {
  constructor(options, styleName, iconProperty, iconSizeProperty, labelSizeProperty, isSymbolizedAsIcon) {
    super(options, styleName);
    (0, _defineProperty2.default)(this, "_iconProperty", void 0);
    (0, _defineProperty2.default)(this, "_iconSizeProperty", void 0);
    (0, _defineProperty2.default)(this, "_labelSizeProperty", void 0);
    (0, _defineProperty2.default)(this, "_isSymbolizedAsIcon", void 0);
    this._iconProperty = iconProperty;
    this._iconSizeProperty = iconSizeProperty;
    this._labelSizeProperty = labelSizeProperty;
    this._isSymbolizedAsIcon = isSymbolizedAsIcon;
  }
  isDisabled() {
    if (this._labelSizeProperty.isDynamic()) {
      // dynamic label size not supported
      return true;
    }
    if (!this._iconSizeProperty.isDynamic() || !this._iconSizeProperty.isComplete()) {
      // icon size is static so there are no concerns with using layout propery and feature-state
      return false;
    }

    // Label position can not be used in concunction with dynamic icon size from joins.
    // Why?
    //   Label position sets layout properties. Layout properties do not support feature-state.
    //   Label position sets a layout property to the interpolate expression from dynamic icon size property
    //   This means that style data for dynamic icon size property can only be retrieved from feature.properties
    //
    return this._isIconSizeFromJoin();
  }
  getDisabledReason() {
    if (this._labelSizeProperty.isDynamic()) {
      return _i18n.i18n.translate('xpack.maps.labelPosition.dynamicLabelSizeNotSupported', {
        defaultMessage: `{labelPositionPropertyLabel} is not supported with '{byValueLabel}' {labelSizePropertyLabel}. Set {labelSizePropertyLabel} to '{fixedLabel}' to enable.`,
        values: {
          byValueLabel: _style_prop_editor.BY_VALUE_LABEL.toLowerCase(),
          fixedLabel: _style_prop_editor.FIXED_LABEL.toLowerCase(),
          labelSizePropertyLabel: (0, _get_vector_style_label.getVectorStyleLabel)(_constants.VECTOR_STYLES.LABEL_SIZE).toLowerCase(),
          labelPositionPropertyLabel: (0, _get_vector_style_label.getVectorStyleLabel)(_constants.VECTOR_STYLES.LABEL_POSITION)
        }
      });
    }
    return this._isIconSizeFromJoin() ? _i18n.i18n.translate('xpack.maps.labelPosition.iconSizeJoinFieldNotSupportMsg', {
      defaultMessage: '{labelPositionPropertyLabel} is not supported with {iconSizePropertyLabel} join field {iconSizeFieldName}. Set {iconSizePropertyLabel} to source field to enable.',
      values: {
        iconSizePropertyLabel: (0, _get_vector_style_label.getVectorStyleLabel)(_constants.VECTOR_STYLES.ICON_SIZE),
        iconSizeFieldName: this._iconSizeProperty.getFieldName(),
        labelPositionPropertyLabel: (0, _get_vector_style_label.getVectorStyleLabel)(_constants.VECTOR_STYLES.LABEL_POSITION)
      }
    }) : '';
  }
  syncLabelPositionWithMb(mbLayerId, mbMap) {
    if (this._options.position === _constants.LABEL_POSITIONS.CENTER || this.isDisabled()) {
      mbMap.setLayoutProperty(mbLayerId, 'text-offset', [0, 0]);
      mbMap.setLayoutProperty(mbLayerId, 'text-anchor', 'center');
      return;
    }
    mbMap.setLayoutProperty(mbLayerId, 'text-anchor', this._options.position === _constants.LABEL_POSITIONS.BOTTOM ? 'top' : 'bottom');
    const labelSize = this._getLabelSize();
    if (this._iconSizeProperty.isDynamic() && this._iconSizeProperty.isComplete() && this._iconSizeProperty.isSizeDynamicConfigComplete()) {
      const dynamicIconSizeOptions = this._iconSizeProperty.getOptions();
      const interpolateExpression = this._iconSizeProperty.getMbSizeExpression({
        forceFeatureProperties: true,
        maxStopOutput: ['literal', this._getTextOffset(dynamicIconSizeOptions.maxSize, labelSize)],
        minStopOutput: ['literal', this._getTextOffset(dynamicIconSizeOptions.minSize, labelSize)]
      });
      mbMap.setLayoutProperty(mbLayerId, 'text-offset', interpolateExpression);
      return;
    }
    const iconSize = !this._iconSizeProperty.isDynamic() ? this._iconSizeProperty.getOptions().size : _vector_style_defaults.DEFAULT_ICON_SIZE;
    mbMap.setLayoutProperty(mbLayerId, 'text-offset', this._getTextOffset(iconSize, labelSize));
  }

  // https://maplibre.org/maplibre-gl-js-docs/style-spec/layers/#layout-symbol-text-offset
  _getTextOffset(symbolSize, labelSize) {
    const ems = symbolSize / labelSize;
    // Positive values indicate right and down, while negative values indicate left and up
    const verticalTextOffset = this._options.position === _constants.LABEL_POSITIONS.BOTTOM ? ems : ems * -1;
    return [0, verticalTextOffset * this._getIconScale()];
  }
  _getIconScale() {
    if (!this._isSymbolizedAsIcon) {
      return 1;
    }
    const iconAnchor = !this._iconProperty.isDynamic() ? this._iconProperty.getSymbolAnchor() : 'center';
    if (iconAnchor === 'center') {
      return 1;
    }

    // using scaling factor of 1.75
    // scaling factor of 1.5 is too small - labels touch top of icon
    // scaling factor of 2 is too big - labels are too far above icon
    return this._options.position === _constants.LABEL_POSITIONS.TOP ? 1.75 : 0;
  }
  _getLabelSize() {
    return !this._labelSizeProperty.isDynamic() ? this._labelSizeProperty.getOptions().size : _vector_style_defaults.DEFAULT_LABEL_SIZE;
  }
  _isIconSizeFromJoin() {
    return this._iconSizeProperty.isDynamic() && this._iconSizeProperty.getFieldOrigin() === _constants.FIELD_ORIGIN.JOIN;
  }
}
exports.LabelPositionProperty = LabelPositionProperty;