"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HeatmapLayer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _layer = require("../layer");
var _heatmap_style = require("../../styles/heatmap/heatmap_style");
var _constants = require("../../../../common/constants");
var _vector_layer = require("../vector_layer");
var _build_vector_request_meta = require("../build_vector_request_meta");
var _tile_meta_feature_utils = require("../../util/tile_meta_feature_utils");
var _mask = require("../vector_layer/mask");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class HeatmapLayer extends _layer.AbstractLayer {
  static createDescriptor(options) {
    const heatmapLayerDescriptor = super.createDescriptor(options);
    heatmapLayerDescriptor.type = _constants.LAYER_TYPE.HEATMAP;
    heatmapLayerDescriptor.style = _heatmap_style.HeatmapStyle.createDescriptor();
    return heatmapLayerDescriptor;
  }
  constructor({
    layerDescriptor,
    source
  }) {
    super({
      layerDescriptor,
      source
    });
    (0, _defineProperty2.default)(this, "_style", void 0);
    if (!layerDescriptor.style) {
      const defaultStyle = _heatmap_style.HeatmapStyle.createDescriptor();
      this._style = new _heatmap_style.HeatmapStyle(defaultStyle);
    } else {
      this._style = new _heatmap_style.HeatmapStyle(layerDescriptor.style);
    }
  }
  _isTiled() {
    // Uses tiled maplibre source 'vector'
    return true;
  }
  getLayerIcon(isTocIcon) {
    const {
      docCount
    } = (0, _tile_meta_feature_utils.getAggsMeta)(this._getTileMetaFeatures());
    return docCount === 0 ? _vector_layer.NO_RESULTS_ICON_AND_TOOLTIPCONTENT : super.getLayerIcon(isTocIcon);
  }
  getSource() {
    return super.getSource();
  }
  getStyleForEditing() {
    return this._style;
  }
  getStyle() {
    return this._style;
  }
  getCurrentStyle() {
    return this._style;
  }
  _getHeatmapLayerId() {
    return this.makeMbLayerId('heatmap');
  }
  getMbLayerIds() {
    return [this._getHeatmapLayerId()];
  }
  ownsMbLayerId(mbLayerId) {
    return this._getHeatmapLayerId() === mbLayerId;
  }
  ownsMbSourceId(mbSourceId) {
    return this.getId() === mbSourceId;
  }
  async syncData(syncContext) {
    await (0, _vector_layer.syncMvtSourceData)({
      buffer: 0,
      hasLabels: false,
      layerId: this.getId(),
      layerName: await this.getDisplayName(),
      prevDataRequest: this.getSourceDataRequest(),
      requestMeta: (0, _build_vector_request_meta.buildVectorRequestMeta)(this.getSource(), [],
      // fieldNames is empty because heatmap layer only support metrics
      syncContext.dataFilters, this.getQuery(), syncContext.isForceRefresh, syncContext.isFeatureEditorOpenForLayer),
      source: this.getSource(),
      syncContext
    });
  }
  _requiresPrevSourceCleanup(mbMap) {
    var _mbSource$tiles;
    const mbSource = mbMap.getSource(this.getMbSourceId());
    if (!mbSource) {
      return false;
    }
    const sourceDataRequest = this.getSourceDataRequest();
    if (!sourceDataRequest) {
      return false;
    }
    const sourceData = sourceDataRequest.getData();
    if (!sourceData) {
      return false;
    }
    return ((_mbSource$tiles = mbSource.tiles) === null || _mbSource$tiles === void 0 ? void 0 : _mbSource$tiles[0]) !== sourceData.tileUrl;
  }
  syncLayerWithMB(mbMap) {
    this._removeStaleMbSourcesAndLayers(mbMap);
    const sourceDataRequest = this.getSourceDataRequest();
    const sourceData = sourceDataRequest ? sourceDataRequest.getData() : undefined;
    if (!sourceData) {
      return;
    }
    const mbSourceId = this.getMbSourceId();
    const mbSource = mbMap.getSource(mbSourceId);
    if (!mbSource) {
      mbMap.addSource(mbSourceId, {
        type: 'vector',
        tiles: [sourceData.tileUrl],
        minzoom: sourceData.tileMinZoom,
        maxzoom: sourceData.tileMaxZoom
      });
    }
    const heatmapLayerId = this._getHeatmapLayerId();
    if (!mbMap.getLayer(heatmapLayerId)) {
      mbMap.addLayer({
        id: heatmapLayerId,
        type: 'heatmap',
        source: mbSourceId,
        ['source-layer']: sourceData.tileSourceLayer,
        paint: {}
      });
    }
    const metricFields = this.getSource().getMetricFields();
    if (!metricFields.length) {
      return;
    }
    const metricField = metricFields[0];

    // do not use tile meta features from previous tile URL to avoid styling new tiles from previous tile meta features
    const tileMetaFeatures = this._requiresPrevSourceCleanup(mbMap) ? [] : this._getTileMetaFeatures();
    let max = 0;
    for (let i = 0; i < tileMetaFeatures.length; i++) {
      const range = metricField.pluckRangeFromTileMetaFeature(tileMetaFeatures[i]);
      if (range) {
        max = Math.max(range.max, max);
      }
    }
    this.getCurrentStyle().setMBPaintProperties({
      mbMap,
      layerId: heatmapLayerId,
      propertyName: metricField.getMbFieldName(),
      max,
      resolution: this.getSource().getGridResolution()
    });
    this.syncVisibilityWithMb(mbMap, heatmapLayerId);
    mbMap.setPaintProperty(heatmapLayerId, 'heatmap-opacity', this.getAlpha());

    // heatmap can implement mask with filter expression because
    // feature-state support is not needed since heatmap layers do not support joins
    const maskDescriptor = metricField.getMask();
    if (maskDescriptor) {
      const mask = new _mask.Mask({
        esAggField: metricField,
        isGeometrySourceMvt: true,
        ...maskDescriptor
      });
      mbMap.setFilter(heatmapLayerId, mask.getMatchUnmaskedExpression());
    }
    mbMap.setLayerZoomRange(heatmapLayerId, this.getMinZoom(), this.getMaxZoom());
  }
  getLayerTypeIconName() {
    return 'heatmap';
  }
  async getFields() {
    return this.getSource().getFields();
  }
  async hasLegendDetails() {
    return true;
  }
  renderLegendDetails() {
    const metricFields = this.getSource().getMetricFields();
    return this.getCurrentStyle().renderLegendDetails(metricFields[0]);
  }
  async getBounds(getDataRequestContext) {
    return await (0, _vector_layer.syncBoundsData)({
      layerId: this.getId(),
      syncContext: getDataRequestContext(this.getId()),
      source: this.getSource(),
      sourceQuery: this.getQuery()
    });
  }
  async isFilteredByGlobalTime() {
    return this.getSource().getApplyGlobalTime() && (await this.getSource().isTimeAware());
  }
  getIndexPatternIds() {
    return this.getSource().getIndexPatternIds();
  }
  getQueryableIndexPatternIds() {
    return this.getSource().getQueryableIndexPatternIds();
  }
  async getLicensedFeatures() {
    return await this.getSource().getLicensedFeatures();
  }
}
exports.HeatmapLayer = HeatmapLayer;