"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateStateContainer = exports.updateControlPanels = exports.subscribeControlGroup = exports.listenUrlChange = exports.initializeFromUrl = exports.initializeControlPanels = exports.getVisibleControlPanelsConfig = void 0;
var _lodash = require("lodash");
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _unifiedDataTable = require("@kbn/unified-data-table");
var _dataset_selection = require("../../../../common/dataset_selection");
var _constants = require("../../../../common/constants");
var _types = require("./types");
var _defaults = require("./defaults");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const listenUrlChange = ({
  stateContainer
}) => context => send => {
  const unsubscribe = stateContainer.appState.subscribe(nextState => {
    const {
      index
    } = nextState;
    const prevIndex = stateContainer.appState.getPrevious().index;

    // Preventing update if the index didn't change
    if (prevIndex === index) return;
    try {
      const datasetSelection = extractDatasetSelectionFromIndex({
        index,
        context
      });
      if ((0, _dataset_selection.isDatasetSelection)(datasetSelection)) {
        send({
          type: 'UPDATE_DATASET_SELECTION',
          data: datasetSelection
        });
      }
    } catch (error) {
      send({
        type: 'DATASET_SELECTION_RESTORE_FAILURE'
      });
    }
  });
  return () => unsubscribe();
};
exports.listenUrlChange = listenUrlChange;
const initializeFromUrl = ({
  stateContainer
}) => async context => {
  const {
    index
  } = stateContainer.appState.getState();
  return extractDatasetSelectionFromIndex({
    index,
    context
  });
};
exports.initializeFromUrl = initializeFromUrl;
const initializeControlPanels = ({
  stateContainer
}) => async context => {
  const urlPanels = stateContainer.stateStorage.get(_defaults.CONTROL_PANELS_URL_KEY);
  const controlPanelsWithId = constructControlPanelsWithDataViewId(stateContainer, urlPanels);
  return controlPanelsWithId;
};
exports.initializeControlPanels = initializeControlPanels;
const extractDatasetSelectionFromIndex = ({
  index,
  context
}) => {
  // If the index parameter doesn't exists, use initialContext value or fallback to AllDatasetSelection
  if (!index) {
    var _context$datasetSelec;
    return (_context$datasetSelec = context.datasetSelection) !== null && _context$datasetSelec !== void 0 ? _context$datasetSelec : _dataset_selection.AllDatasetSelection.create();
  }
  const rawDatasetSelection = (0, _dataset_selection.decodeDatasetSelectionId)(index);
  const datasetSelection = (0, _dataset_selection.hydrateDatasetSelection)(rawDatasetSelection);
  return datasetSelection;
};
const subscribeControlGroup = ({
  stateContainer
}) => context => send => {
  if (!('controlGroupAPI' in context)) return;
  const filtersSubscription = context.controlGroupAPI.onFiltersPublished$.subscribe(newFilters => {
    stateContainer.internalState.transitions.setCustomFilters(newFilters);
    stateContainer.actions.fetchData();
  });

  // Keeps our state in sync with the url changes and makes sure it adheres to correct schema
  const urlSubscription = stateContainer.stateStorage.change$(_defaults.CONTROL_PANELS_URL_KEY).subscribe(controlPanels => {
    if (!(0, _fastDeepEqual.default)(controlPanels, context.controlPanels)) {
      send({
        type: 'UPDATE_CONTROL_PANELS',
        controlPanels
      });
    }
  });

  // Keeps the url in sync with the controls state after change
  const inputSubscription = context.controlGroupAPI.getInput$().subscribe(({
    panels
  }) => {
    if (!(0, _fastDeepEqual.default)(panels, context.controlPanels)) {
      send({
        type: 'UPDATE_CONTROL_PANELS',
        controlPanels: panels
      });
    }
  });
  return () => {
    filtersSubscription.unsubscribe();
    urlSubscription.unsubscribe();
    inputSubscription.unsubscribe();
  };
};
exports.subscribeControlGroup = subscribeControlGroup;
const updateControlPanels = ({
  stateContainer
}) => async (context, event) => {
  if (!('controlGroupAPI' in context)) return;
  const newControlPanels = 'controlPanels' in event && event.controlPanels || context.controlPanels;
  const controlPanelsWithId = constructControlPanelsWithDataViewId(stateContainer, newControlPanels);
  context.controlGroupAPI.updateInput({
    panels: controlPanelsWithId
  });
  return controlPanelsWithId;
};
exports.updateControlPanels = updateControlPanels;
const updateStateContainer = ({
  stateContainer
}) => async () => {
  var _grid$columns;
  const {
    breakdownField,
    columns,
    grid,
    rowHeight
  } = stateContainer.appState.getState();
  const stateUpdates = {};

  // Update data grid columns list
  const shouldSetDefaultColumns = stateContainer.appState.isEmptyURL() || !columns || columns.length === 0;
  if (shouldSetDefaultColumns) {
    stateUpdates.columns = _constants.DATA_GRID_DEFAULT_COLUMNS;
  }

  // Configure DataGrid columns preferences
  const initialColumnsPreferences = (_grid$columns = grid === null || grid === void 0 ? void 0 : grid.columns) !== null && _grid$columns !== void 0 ? _grid$columns : {};
  stateUpdates.grid = {
    columns: {
      ..._constants.DATA_GRID_COLUMNS_PREFERENCES,
      ...initialColumnsPreferences
    }
  };

  // Configure rowHeight preference
  stateUpdates.rowHeight = rowHeight !== null && rowHeight !== void 0 ? rowHeight : _unifiedDataTable.ROWS_HEIGHT_OPTIONS.single;

  // Configure breakdown field preference
  stateUpdates.breakdownField = breakdownField !== null && breakdownField !== void 0 ? breakdownField : _constants.LOG_LEVEL_FIELD;

  // Finally batch update state app state
  stateContainer.appState.update(stateUpdates, true);
};

/**
 * Utils
 */
exports.updateStateContainer = updateStateContainer;
const constructControlPanelsWithDataViewId = (stateContainer, newControlPanels) => {
  const dataView = stateContainer.internalState.getState().dataView;
  const validatedControlPanels = isValidState(newControlPanels) ? newControlPanels : getVisibleControlPanelsConfig(dataView);
  const controlsPanelsWithId = mergeDefaultPanelsWithUrlConfig(dataView, validatedControlPanels);
  if (!(0, _fastDeepEqual.default)(controlsPanelsWithId, stateContainer.stateStorage.get(_defaults.CONTROL_PANELS_URL_KEY))) {
    stateContainer.stateStorage.set(_defaults.CONTROL_PANELS_URL_KEY, cleanControlPanels(controlsPanelsWithId), {
      replace: true
    });
  }
  return controlsPanelsWithId;
};
const isValidState = state => {
  return Object.keys(state !== null && state !== void 0 ? state : {}).length > 0 && _types.ControlPanelRT.is(state);
};
const getVisibleControlPanels = dataView => _defaults.availableControlPanelFields.filter(panelKey => (dataView === null || dataView === void 0 ? void 0 : dataView.fields.getByName(panelKey)) !== undefined);
const getVisibleControlPanelsConfig = dataView => {
  return getVisibleControlPanels(dataView).reduce((panelsMap, panelKey) => {
    const config = _defaults.controlPanelConfigs[panelKey];
    return {
      ...panelsMap,
      [panelKey]: config
    };
  }, {});
};
exports.getVisibleControlPanelsConfig = getVisibleControlPanelsConfig;
const addDataViewIdToControlPanels = (controlPanels, dataViewId = '') => {
  return (0, _lodash.mapValues)(controlPanels, controlPanelConfig => ({
    ...controlPanelConfig,
    explicitInput: {
      ...controlPanelConfig.explicitInput,
      dataViewId
    }
  }));
};
const cleanControlPanels = controlPanels => {
  return (0, _lodash.mapValues)(controlPanels, controlPanelConfig => {
    const {
      explicitInput
    } = controlPanelConfig;
    const {
      dataViewId,
      ...rest
    } = explicitInput;
    return {
      ...controlPanelConfig,
      explicitInput: rest
    };
  });
};
const mergeDefaultPanelsWithUrlConfig = (dataView, urlPanels) => {
  // Get default panel configs from existing fields in data view
  const visiblePanels = getVisibleControlPanelsConfig(dataView);

  // Get list of panel which can be overridden to avoid merging additional config from url
  const existingKeys = Object.keys(visiblePanels);
  const controlPanelsToOverride = (0, _lodash.pick)(urlPanels, existingKeys);

  // Merge default and existing configs and add dataView.id to each of them
  return addDataViewIdToControlPanels({
    ...visiblePanels,
    ...controlPanelsToOverride
  }, dataView.id);
};