"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPureDataViewsStateMachine = exports.createDataViewsStateMachine = void 0;
var _lodash = require("lodash");
var _xstate = require("xstate");
var _parse_data_view_list_item = require("../../../utils/parse_data_view_list_item");
var _comparator_by_field = require("../../../utils/comparator_by_field");
var _defaults = require("./defaults");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createPureDataViewsStateMachine = (initialContext = _defaults.DEFAULT_CONTEXT) => (0, _xstate.createMachine)({
  /** @xstate-layout N4IgpgJg5mDOIC5QBECGAXVA1AlmA7rAHQCuAdjhejqgDY4BekAxADIDyAgsgPrKcAVTjywBJAKIB1AMoBtAAwBdRKAAOAe1g5q6sipAAPRABYAzKaIB2a6bOWArABoQAT0QBaAJzyi9y8YAmAEYADgcAX3DnNExcAmJadVQISihmCF0wIkoAN3UAayyY7DxCIkTk1IRc9QBjDBxdBUVm-Q0tHT0kQxNjSyJPP3lTIICnV0RLTyJ5eUsQ0ftI6IwS+PKklLI0sAAnXfVdolVaDAAzQ4BbImK4soqtqGqyPPrO5tbu9u1GrtAjBAANgcRFMDmcbgQIPsIQCw0WyxAt1KCU2kGyEFoYGY0nEnAASgBhAASfEEwjEUjkSjamh+un0AOsgN8nksgPsgLGEJMHKIgNCgRC9mMC2siOR6we6JwmOx0nY+IEZKEIgkMk+ajpnUZJk8ISIAQCHK540hpgCBoWgOM8lNEtWd1RyRlcpx4lY4kJyv4qspkk1IG+Ou6AMB8gNfkCoXBEwQ9gtM1GgzZ+qC4qiSMdKI2LogRAgYAARupyLUwNIwKhdrUABapSWEd0Ekkqinq6nKL7a366hBBeT2ez8zncuMJ6ZzEX2VNiywO2I56X5wslssVqs1+vbRuwZgGWCYdBZVBnY+7AAUplmAEpmLvc4WV8XS2Ry5Xq3WG9n4oHg73Q0mEIDXDYwwLMexLUBMEeQQcwLE8II0wCYxARCcMpwXNZ7k2VIADFUBwLEIGYfEPS4XhfXbKk-x7BlAKBaDfACSxbDBM0PACUFPGMBMEURMh1ELeBul3WkOgA-5EFMYUrBsOwOIQdwUK4oYRjGLCnVICgqBoegmAgcT6T+HoEDAlkxnmRZYPcIIhxmSx4Q0zMHweVIjJDKSzNtIgQlmKyx0hZThxCHi+OclZFylNFDO7CT6K83ifCNE1AsQFCglBIIbTtCKsyinC8wxLEPMk0zARtUF7Ds01YN44dAj8OFglFBTNKXGKCxfdcPy3b8CpErV4pMgEBztUEVJjRSzEyzxLOaoJWvY9rosqbYCKIyBSoS0y5q4nLarjZT+j8BZnMiIA */
  context: initialContext,
  preserveActionOrder: true,
  predictableActionArguments: true,
  id: 'DataViews',
  initial: 'uninitialized',
  states: {
    uninitialized: {
      on: {
        LOAD_DATA_VIEWS: 'loading'
      }
    },
    loading: {
      id: 'loading',
      invoke: {
        src: 'loadDataViews',
        onDone: {
          target: 'loaded',
          actions: ['storeInCache', 'storeDataViews', 'storeSearch']
        },
        onError: 'loadingFailed'
      }
    },
    loaded: {
      id: 'loaded',
      initial: 'idle',
      states: {
        idle: {
          on: {
            SEARCH_DATA_VIEWS: 'debounceSearchingDataViews',
            SORT_DATA_VIEWS: {
              actions: ['storeSearch', 'searchDataViews']
            },
            SELECT_DATA_VIEW: {
              actions: ['navigateToDiscoverDataView']
            }
          }
        },
        debounceSearchingDataViews: {
          entry: 'storeSearch',
          on: {
            SEARCH_DATA_VIEWS: 'debounceSearchingDataViews'
          },
          after: {
            300: {
              target: 'idle',
              actions: 'searchDataViews'
            }
          }
        }
      }
    },
    loadingFailed: {
      entry: ['clearCache', 'clearData', 'storeError'],
      exit: 'clearError',
      on: {
        RELOAD_DATA_VIEWS: 'loading'
      }
    }
  }
}, {
  actions: {
    storeSearch: (0, _xstate.assign)((_context, event) => ({
      // Store search from search event
      ...('search' in event && {
        search: event.search
      })
    })),
    storeDataViews: (0, _xstate.assign)((_context, event) => 'data' in event && !(0, _lodash.isError)(event.data) ? {
      dataViewsSource: event.data,
      dataViews: event.data
    } : {}),
    searchDataViews: (0, _xstate.assign)(context => {
      if (context.dataViewsSource !== null) {
        return {
          dataViews: searchDataViews(context.dataViewsSource, context.search)
        };
      }
      return {};
    }),
    storeInCache: (context, event) => {
      if ('data' in event && !(0, _lodash.isError)(event.data)) {
        context.cache.set(context.search, event.data);
      }
    },
    storeError: (0, _xstate.assign)((_context, event) => 'data' in event && (0, _lodash.isError)(event.data) ? {
      error: event.data
    } : {}),
    clearCache: context => {
      context.cache.reset();
    },
    clearData: (0, _xstate.assign)(_context => ({
      dataViews: null
    })),
    clearError: (0, _xstate.assign)(_context => ({
      error: null
    }))
  }
});
exports.createPureDataViewsStateMachine = createPureDataViewsStateMachine;
const createDataViewsStateMachine = ({
  initialContext,
  dataViews,
  discover
}) => createPureDataViewsStateMachine(initialContext).withConfig({
  actions: {
    navigateToDiscoverDataView: (_context, event) => {
      if (event.type === 'SELECT_DATA_VIEW' && 'dataView' in event) {
        var _discover$locator;
        (_discover$locator = discover.locator) === null || _discover$locator === void 0 ? void 0 : _discover$locator.navigate({
          dataViewId: event.dataView.id
        });
      }
    }
  },
  services: {
    loadDataViews: context => {
      const searchParams = context.search;
      return context.cache.has(searchParams) ? Promise.resolve(context.cache.get(searchParams)) : dataViews.getIdsWithTitle().then(views => views.map(_parse_data_view_list_item.parseDataViewListItem)).then(views => searchDataViews(views, searchParams));
    }
  }
});
exports.createDataViewsStateMachine = createDataViewsStateMachine;
const searchDataViews = (dataViews, search) => {
  const {
    name,
    sortOrder
  } = search;
  return dataViews.filter(dataView => {
    var _dataView$name;
    return Boolean((_dataView$name = dataView.name) === null || _dataView$name === void 0 ? void 0 : _dataView$name.includes(name !== null && name !== void 0 ? name : ''));
  }).sort((0, _comparator_by_field.createComparatorByField)('name', sortOrder));
};