"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPersisted = void 0;
exports.loadInitial = loadInitial;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _ = require("..");
var _lens_slice = require("../lens_slice");
var _utils = require("../../utils");
var _editor_frame = require("../../editor_frame_service/editor_frame");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getPersisted = async ({
  initialInput,
  lensServices,
  history
}) => {
  const {
    notifications,
    spaces,
    attributeService
  } = lensServices;
  let doc;
  try {
    const result = await attributeService.unwrapAttributes(initialInput);
    if (!result) {
      return {
        doc: {
          ...initialInput,
          type: _constants.LENS_EMBEDDABLE_TYPE
        },
        sharingSavedObjectProps: {
          outcome: 'exactMatch'
        }
      };
    }
    const {
      metaInfo,
      attributes
    } = result;
    const sharingSavedObjectProps = metaInfo === null || metaInfo === void 0 ? void 0 : metaInfo.sharingSavedObjectProps;
    if (spaces && (sharingSavedObjectProps === null || sharingSavedObjectProps === void 0 ? void 0 : sharingSavedObjectProps.outcome) === 'aliasMatch' && history) {
      // We found this object by a legacy URL alias from its old ID; redirect the user to the page with its new ID, preserving any URL hash
      const newObjectId = sharingSavedObjectProps.aliasTargetId; // This is always defined if outcome === 'aliasMatch'
      const newPath = lensServices.http.basePath.prepend(`${(0, _constants.getEditPath)(newObjectId)}${history.location.search}`);
      await spaces.ui.redirectLegacyUrl({
        path: newPath,
        aliasPurpose: sharingSavedObjectProps.aliasPurpose,
        objectNoun: _i18n.i18n.translate('xpack.lens.legacyUrlConflict.objectNoun', {
          defaultMessage: 'Lens visualization'
        })
      });
    }
    doc = {
      ...initialInput,
      ...attributes,
      type: _constants.LENS_EMBEDDABLE_TYPE
    };
    return {
      doc,
      sharingSavedObjectProps: {
        aliasTargetId: sharingSavedObjectProps === null || sharingSavedObjectProps === void 0 ? void 0 : sharingSavedObjectProps.aliasTargetId,
        outcome: sharingSavedObjectProps === null || sharingSavedObjectProps === void 0 ? void 0 : sharingSavedObjectProps.outcome
      }
    };
  } catch (e) {
    notifications.toasts.addDanger(_i18n.i18n.translate('xpack.lens.app.docLoadingError', {
      defaultMessage: 'Error loading saved document'
    }));
  }
};
exports.getPersisted = getPersisted;
function loadInitial(store, storeDeps, {
  redirectCallback,
  initialInput,
  history,
  inlineEditing
}, autoApplyDisabled) {
  var _lens$persistedDoc2;
  const {
    lensServices,
    datasourceMap,
    initialContext,
    initialStateFromLocator,
    visualizationMap
  } = storeDeps;
  const {
    resolvedDateRange,
    searchSessionId,
    isLinkedToOriginatingApp,
    ...emptyState
  } = (0, _lens_slice.getPreloadedState)(storeDeps);
  const {
    attributeService,
    notifications,
    data,
    dashboardFeatureFlag
  } = lensServices;
  const {
    lens
  } = store.getState();
  const loaderSharedArgs = {
    dataViews: lensServices.dataViews,
    storage: lensServices.storage,
    eventAnnotationService: lensServices.eventAnnotationService,
    defaultIndexPatternId: lensServices.uiSettings.get('defaultIndex')
  };
  let activeDatasourceId;
  if (initialContext && 'query' in initialContext) {
    activeDatasourceId = 'textBased';
  }
  if (initialStateFromLocator) {
    const newFilters = initialStateFromLocator.filters ? (0, _lodash.cloneDeep)(initialStateFromLocator.filters) : undefined;
    if (newFilters) {
      data.query.filterManager.setAppFilters(newFilters);
    }
    if (initialStateFromLocator.resolvedDateRange) {
      const newTimeRange = {
        from: initialStateFromLocator.resolvedDateRange.fromDate,
        to: initialStateFromLocator.resolvedDateRange.toDate
      };
      data.query.timefilter.timefilter.setTime(newTimeRange);
    }
    // URL Reporting is using the locator params but also passing the savedObjectId
    // so be sure to not go here as there's no full snapshot URL
    if (!initialInput) {
      var _lens$persistedDoc;
      const locatorReferences = 'references' in initialStateFromLocator ? initialStateFromLocator.references : undefined;
      return (0, _editor_frame.initializeSources)({
        datasourceMap,
        visualizationMap,
        visualizationState: emptyState.visualization,
        datasourceStates: emptyState.datasourceStates,
        initialContext,
        adHocDataViews: ((_lens$persistedDoc = lens.persistedDoc) === null || _lens$persistedDoc === void 0 ? void 0 : _lens$persistedDoc.state.adHocDataViews) || initialStateFromLocator.dataViewSpecs,
        references: locatorReferences,
        ...loaderSharedArgs
      }, {
        isFullEditor: true
      }).then(({
        datasourceStates,
        visualizationState,
        indexPatterns,
        indexPatternRefs,
        annotationGroups
      }) => {
        const currentSessionId = (initialStateFromLocator === null || initialStateFromLocator === void 0 ? void 0 : initialStateFromLocator.searchSessionId) || data.search.session.getSessionId();
        store.dispatch((0, _.initExisting)({
          isSaveable: true,
          filters: initialStateFromLocator.filters || data.query.filterManager.getFilters(),
          query: initialStateFromLocator.query || emptyState.query,
          searchSessionId: currentSessionId,
          activeDatasourceId: emptyState.activeDatasourceId,
          visualization: {
            activeId: emptyState.visualization.activeId,
            state: visualizationState
          },
          dataViews: (0, _utils.getInitialDataViewsObject)(indexPatterns, indexPatternRefs),
          datasourceStates: Object.entries(datasourceStates).reduce((state, [datasourceId, datasourceState]) => ({
            ...state,
            [datasourceId]: {
              ...datasourceState,
              isLoading: false
            }
          }), {}),
          isLoading: false,
          annotationGroups
        }));
        if (autoApplyDisabled) {
          store.dispatch((0, _lens_slice.disableAutoApply)());
        }
      }).catch(e => {
        notifications.toasts.addDanger({
          title: e.message
        });
      });
    }
  }
  if (!initialInput || attributeService.inputIsRefType(initialInput) && initialInput.savedObjectId === ((_lens$persistedDoc2 = lens.persistedDoc) === null || _lens$persistedDoc2 === void 0 ? void 0 : _lens$persistedDoc2.savedObjectId)) {
    var _lens$persistedDoc3;
    const newFilters = initialContext && 'searchFilters' in initialContext && initialContext.searchFilters ? (0, _lodash.cloneDeep)(initialContext.searchFilters) : undefined;
    if (newFilters) {
      data.query.filterManager.setAppFilters(newFilters);
    }
    return (0, _editor_frame.initializeSources)({
      datasourceMap,
      visualizationMap,
      visualizationState: lens.visualization,
      datasourceStates: lens.datasourceStates,
      initialContext,
      adHocDataViews: (_lens$persistedDoc3 = lens.persistedDoc) === null || _lens$persistedDoc3 === void 0 ? void 0 : _lens$persistedDoc3.state.adHocDataViews,
      ...loaderSharedArgs
    }, {
      isFullEditor: true
    }).then(({
      datasourceStates,
      indexPatterns,
      indexPatternRefs
    }) => {
      store.dispatch((0, _.initEmpty)({
        newState: {
          ...emptyState,
          dataViews: (0, _utils.getInitialDataViewsObject)(indexPatterns, indexPatternRefs),
          searchSessionId: data.search.session.getSessionId() || data.search.session.start(),
          ...(activeDatasourceId && {
            activeDatasourceId
          }),
          datasourceStates: Object.entries(datasourceStates).reduce((state, [datasourceId, datasourceState]) => ({
            ...state,
            [datasourceId]: {
              ...datasourceState,
              isLoading: false
            }
          }), {}),
          isLoading: false
        },
        initialContext
      }));
      if (autoApplyDisabled) {
        store.dispatch((0, _lens_slice.disableAutoApply)());
      }
    }).catch(e => {
      notifications.toasts.addDanger({
        title: e.message
      });
      redirectCallback === null || redirectCallback === void 0 ? void 0 : redirectCallback();
    });
  }
  return getPersisted({
    initialInput,
    lensServices,
    history
  }).then(persisted => {
    if (persisted) {
      const {
        doc,
        sharingSavedObjectProps
      } = persisted;
      if (attributeService.inputIsRefType(initialInput)) {
        lensServices.chrome.recentlyAccessed.add((0, _constants.getFullPath)(initialInput.savedObjectId), doc.title, initialInput.savedObjectId);
      }
      const docDatasourceStates = Object.entries(doc.state.datasourceStates).reduce((stateMap, [datasourceId, datasourceState]) => ({
        ...stateMap,
        [datasourceId]: {
          isLoading: true,
          state: datasourceState
        }
      }), {});

      // when the embeddable is initialized from the dashboard we don't want to inject the filters
      // as this will replace the parent application filters (such as a dashboard)
      if (!Boolean(inlineEditing)) {
        const filters = data.query.filterManager.inject(doc.state.filters, doc.references);
        // Don't overwrite any pinned filters
        data.query.filterManager.setAppFilters(filters);
      }
      const docVisualizationState = {
        activeId: doc.visualizationType,
        state: doc.state.visualization
      };
      return (0, _editor_frame.initializeSources)({
        datasourceMap,
        visualizationMap,
        visualizationState: docVisualizationState,
        datasourceStates: docDatasourceStates,
        references: [...doc.references, ...(doc.state.internalReferences || [])],
        initialContext,
        dataViews: lensServices.dataViews,
        eventAnnotationService: lensServices.eventAnnotationService,
        storage: lensServices.storage,
        adHocDataViews: doc.state.adHocDataViews,
        defaultIndexPatternId: lensServices.uiSettings.get('defaultIndex')
      }, {
        isFullEditor: true
      }).then(({
        datasourceStates,
        visualizationState,
        indexPatterns,
        indexPatternRefs,
        annotationGroups
      }) => {
        const currentSessionId = data.search.session.getSessionId();
        store.dispatch((0, _.initExisting)({
          isSaveable: true,
          sharingSavedObjectProps,
          filters: data.query.filterManager.getFilters(),
          query: doc.state.query,
          searchSessionId: dashboardFeatureFlag.allowByValueEmbeddables && !(initialInput !== null && initialInput !== void 0 && initialInput.savedObjectId) && currentSessionId ? currentSessionId : data.search.session.start(),
          persistedDoc: doc,
          activeDatasourceId: (0, _utils.getInitialDatasourceId)(datasourceMap, doc),
          visualization: {
            activeId: doc.visualizationType,
            state: visualizationState
          },
          dataViews: (0, _utils.getInitialDataViewsObject)(indexPatterns, indexPatternRefs),
          datasourceStates: Object.entries(datasourceStates).reduce((state, [datasourceId, datasourceState]) => ({
            ...state,
            [datasourceId]: {
              ...datasourceState,
              isLoading: false
            }
          }), {}),
          isLoading: false,
          annotationGroups
        }));
        if (autoApplyDisabled) {
          store.dispatch((0, _lens_slice.disableAutoApply)());
        }
      }).catch(e => notifications.toasts.addDanger({
        title: e.message
      }));
    } else {
      redirectCallback === null || redirectCallback === void 0 ? void 0 : redirectCallback();
    }
  }, () => {
    store.dispatch((0, _.setState)({
      isLoading: false
    }));
    redirectCallback === null || redirectCallback === void 0 ? void 0 : redirectCallback();
  }).catch(e => {
    notifications.toasts.addDanger({
      title: e.message
    });
    redirectCallback === null || redirectCallback === void 0 ? void 0 : redirectCallback();
  });
}