"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.timeScaleOperation = void 0;
var _i18n = require("@kbn/i18n");
var _utils = require("./utils");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const timeScaleOperation = {
  type: 'normalize_by_unit',
  priority: 1,
  displayName: _i18n.i18n.translate('xpack.lens.indexPattern.timeScale', {
    defaultMessage: 'Normalize by unit'
  }),
  input: 'fullReference',
  selectionStyle: 'hidden',
  requiredReferences: [{
    input: ['field', 'managedReference', 'fullReference'],
    validateMetadata: meta => meta.dataType === 'number' && !meta.isBucketed
  }],
  operationParams: [{
    name: 'unit',
    type: 'string',
    required: true
  }],
  getPossibleOperation: () => {
    return {
      dataType: 'number',
      isBucketed: false,
      scale: 'ratio'
    };
  },
  getDefaultLabel: (column, columns, indexPattern) => {
    return 'normalize_by_unit';
  },
  toExpression: (layer, columnId) => {
    const currentColumn = layer.columns[columnId];
    const buckets = layer.columnOrder.filter(colId => layer.columns[colId].isBucketed);
    const dateColumn = buckets.find(colId => layer.columns[colId].operationType === 'date_histogram');
    return [{
      type: 'function',
      function: 'lens_time_scale',
      arguments: {
        dateColumnId: dateColumn ? [dateColumn] : [],
        inputColumnId: [currentColumn.references[0]],
        outputColumnId: [columnId],
        outputColumnName: [currentColumn.label],
        targetUnit: [currentColumn.params.unit],
        reducedTimeRange: currentColumn.reducedTimeRange ? [currentColumn.reducedTimeRange] : []
      }
    }];
  },
  buildColumn: ({
    referenceIds,
    previousColumn,
    layer,
    indexPattern
  }, columnParams) => {
    return {
      label: 'Normalize by unit',
      dataType: 'number',
      operationType: 'normalize_by_unit',
      isBucketed: false,
      scale: 'ratio',
      references: referenceIds,
      params: {
        ...(0, _helpers.getFormatFromPreviousColumn)(previousColumn),
        unit: columnParams === null || columnParams === void 0 ? void 0 : columnParams.unit
      }
    };
  },
  isTransferable: () => {
    return true;
  },
  getErrorMessage: (layer, columnId) => {
    return (0, _helpers.combineErrorMessages)([(0, _utils.getErrorsForDateReference)(layer, columnId, _i18n.i18n.translate('xpack.lens.indexPattern.timeScale', {
      defaultMessage: 'Normalize by unit'
    })), !layer.columns[columnId].params.unit ? [_i18n.i18n.translate('xpack.lens.indexPattern.timeScale.missingUnit', {
      defaultMessage: 'No unit specified for normalize by unit.'
    })] : [], ['s', 'm', 'h', 'd'].indexOf(layer.columns[columnId].params.unit || 's') === -1 ? [_i18n.i18n.translate('xpack.lens.indexPattern.timeScale.wrongUnit', {
      defaultMessage: 'Unknown unit specified: use s, m, h or d.'
    })] : []]);
  },
  filterable: false,
  shiftable: false,
  documentation: {
    section: 'calculation',
    signature: _i18n.i18n.translate('xpack.lens.indexPattern.time_scale', {
      defaultMessage: 'metric: number, unit: s|m|h|d|w|M|y'
    }),
    description: _i18n.i18n.translate('xpack.lens.indexPattern.time_scale.documentation.markdown', {
      defaultMessage: `

This advanced function is useful for normalizing counts and sums to a specific time interval. It allows for integration with metrics that are stored already normalized to a specific time interval.

This function can only be used if there's a date histogram function used in the current chart.

Example: A ratio comparing an already normalized metric to another metric that needs to be normalized.
\`normalize_by_unit(counter_rate(max(system.diskio.write.bytes)), unit='s') / last_value(apache.status.bytes_per_second)\`
      `
    })
  }
};
exports.timeScaleOperation = timeScaleOperation;