"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildMetric = buildMetric;
var _types = require("../types");
var _utils = require("../utils");
var _columns = require("../columns");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const ACCESSOR = 'metric_formula_accessor';
const HISTOGRAM_COLUMN_NAME = 'x_date_histogram';
const TRENDLINE_LAYER_ID = `layer_trendline`;
function getAccessorName(type) {
  return `${ACCESSOR}_${type}`;
}
function buildVisualizationState(config) {
  const layer = config;
  return {
    layerId: _types.DEFAULT_LAYER_ID,
    layerType: 'data',
    metricAccessor: ACCESSOR,
    color: layer.seriesColor,
    // subtitle: layer.subtitle,
    showBar: false,
    ...(layer.querySecondaryMetric ? {
      secondaryMetricAccessor: getAccessorName('secondary')
    } : {}),
    ...(layer.queryMaxValue ? {
      maxAccessor: getAccessorName('max'),
      showBar: true
    } : {}),
    ...(layer.breakdown ? {
      breakdownByAccessor: getAccessorName('breakdown')
    } : {}),
    ...(layer.trendLine ? {
      trendlineLayerId: `${_types.DEFAULT_LAYER_ID}_trendline`,
      trendlineLayerType: 'metricTrendline',
      trendlineMetricAccessor: `${ACCESSOR}_trendline`,
      trendlineTimeAccessor: HISTOGRAM_COLUMN_NAME,
      ...(layer.querySecondaryMetric ? {
        trendlineSecondaryMetricAccessor: `${ACCESSOR}_secondary_trendline`
      } : {}),
      ...(layer.queryMaxValue ? {
        trendlineMaxAccessor: `${ACCESSOR}_max_trendline`
      } : {}),
      ...(layer.breakdown ? {
        trendlineBreakdownByAccessor: `${ACCESSOR}_breakdown_trendline`
      } : {})
    } : {})
  };
}
function buildFormulaLayer(layer, i, dataView, formulaAPI) {
  const baseLayer = {
    columnOrder: [ACCESSOR, HISTOGRAM_COLUMN_NAME],
    columns: {
      [HISTOGRAM_COLUMN_NAME]: (0, _columns.getHistogramColumn)({
        options: {
          sourceField: dataView.timeFieldName,
          params: {
            interval: 'auto',
            includeEmptyRows: true
          }
        }
      })
    },
    sampling: 1
  };
  const layers = {
    [_types.DEFAULT_LAYER_ID]: {
      ...(0, _columns.getFormulaColumn)(ACCESSOR, {
        value: layer.value
      }, dataView, formulaAPI)
    },
    ...(layer.trendLine ? {
      [TRENDLINE_LAYER_ID]: {
        linkToLayers: [_types.DEFAULT_LAYER_ID],
        ...(0, _columns.getFormulaColumn)(`${ACCESSOR}_trendline`, {
          value: layer.value
        }, dataView, formulaAPI, baseLayer)
      }
    } : {})
  };
  const defaultLayer = layers[_types.DEFAULT_LAYER_ID];
  const trendLineLayer = layers[TRENDLINE_LAYER_ID];
  if (layer.breakdown) {
    const columnName = getAccessorName('breakdown');
    const breakdownColumn = (0, _columns.getBreakdownColumn)({
      options: layer.breakdown,
      dataView
    });
    (0, _utils.addLayerColumn)(defaultLayer, columnName, breakdownColumn, true);
    if (trendLineLayer) {
      (0, _utils.addLayerColumn)(trendLineLayer, `${columnName}_trendline`, breakdownColumn, true);
    }
  }
  if (layer.querySecondaryMetric) {
    const columnName = getAccessorName('secondary');
    const formulaColumn = (0, _columns.getFormulaColumn)(columnName, {
      value: layer.querySecondaryMetric
    }, dataView, formulaAPI);
    (0, _utils.addLayerFormulaColumns)(defaultLayer, formulaColumn);
    if (trendLineLayer) {
      (0, _utils.addLayerFormulaColumns)(trendLineLayer, formulaColumn, 'X0');
    }
  }
  if (layer.queryMaxValue) {
    const columnName = getAccessorName('max');
    const formulaColumn = (0, _columns.getFormulaColumn)(columnName, {
      value: layer.queryMaxValue
    }, dataView, formulaAPI);
    (0, _utils.addLayerFormulaColumns)(defaultLayer, formulaColumn);
    if (trendLineLayer) {
      (0, _utils.addLayerFormulaColumns)(trendLineLayer, formulaColumn, 'X0');
    }
  }
  return layers[_types.DEFAULT_LAYER_ID];
}
function getValueColumns(layer) {
  if (layer.breakdown && typeof layer.breakdown !== 'string') {
    throw new Error('breakdown must be a field name when not using index source');
  }
  return [...(layer.breakdown ? [(0, _columns.getValueColumn)(getAccessorName('breakdown'), layer.breakdown)] : []), (0, _columns.getValueColumn)(ACCESSOR, layer.value), ...(layer.queryMaxValue ? [(0, _columns.getValueColumn)(getAccessorName('max'), layer.queryMaxValue)] : []), ...(layer.querySecondaryMetric ? [(0, _columns.getValueColumn)(getAccessorName('secondary'), layer.querySecondaryMetric)] : [])];
}
async function buildMetric(config, {
  dataViewsAPI,
  formulaAPI
}) {
  const dataviews = {};
  const _buildFormulaLayer = (cfg, i, dataView) => buildFormulaLayer(cfg, i, dataView, formulaAPI);
  const datasourceStates = await (0, _utils.buildDatasourceStates)(config, dataviews, _buildFormulaLayer, getValueColumns, dataViewsAPI);
  return {
    title: config.title,
    visualizationType: 'lnsMetric',
    references: (0, _utils.buildReferences)(dataviews),
    state: {
      datasourceStates,
      internalReferences: [],
      filters: [],
      query: {
        language: 'kuery',
        query: ''
      },
      visualization: buildVisualizationState(config),
      // Getting the spec from a data view is a heavy operation, that's why the result is cached.
      adHocDataViews: (0, _utils.getAdhocDataviews)(dataviews)
    }
  };
}