"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InventoryViewsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _inventory_views = require("../../../common/inventory_views");
var _runtime_types = require("../../../common/runtime_types");
var _inventory_view = require("../../saved_objects/inventory_view");
var _types = require("../../saved_objects/inventory_view/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class InventoryViewsClient {
  constructor(logger, savedObjectsClient, infraSources) {
    this.logger = logger;
    this.savedObjectsClient = savedObjectsClient;
    this.infraSources = infraSources;
  }
  async find(query) {
    var _query$sourceId;
    this.logger.debug('Trying to load inventory views ...');
    const sourceId = (_query$sourceId = query.sourceId) !== null && _query$sourceId !== void 0 ? _query$sourceId : InventoryViewsClient.DEFAULT_SOURCE_ID;
    const [sourceConfiguration, inventoryViewSavedObject] = await Promise.all([this.infraSources.getSourceConfiguration(this.savedObjectsClient, sourceId), this.getAllViews()]);
    const defaultView = InventoryViewsClient.createStaticView(sourceConfiguration.configuration.inventoryDefaultView);
    const views = inventoryViewSavedObject.saved_objects.map(savedObject => this.mapSavedObjectToInventoryView(savedObject, sourceConfiguration.configuration.inventoryDefaultView));
    const inventoryViews = [defaultView, ...views];
    const sortedInventoryViews = this.moveDefaultViewOnTop(inventoryViews);
    return sortedInventoryViews;
  }
  async get(inventoryViewId, query) {
    var _query$sourceId2;
    this.logger.debug(`Trying to load inventory view with id ${inventoryViewId} ...`);
    const sourceId = (_query$sourceId2 = query.sourceId) !== null && _query$sourceId2 !== void 0 ? _query$sourceId2 : InventoryViewsClient.DEFAULT_SOURCE_ID;

    // Handle the case where the requested resource is the static inventory view
    if (inventoryViewId === InventoryViewsClient.STATIC_VIEW_ID) {
      const sourceConfiguration = await this.infraSources.getSourceConfiguration(this.savedObjectsClient, sourceId);
      return InventoryViewsClient.createStaticView(sourceConfiguration.configuration.inventoryDefaultView);
    }
    const [sourceConfiguration, inventoryViewSavedObject] = await Promise.all([this.infraSources.getSourceConfiguration(this.savedObjectsClient, sourceId), this.savedObjectsClient.get(_inventory_view.inventoryViewSavedObjectName, inventoryViewId)]);
    return this.mapSavedObjectToInventoryView(inventoryViewSavedObject, sourceConfiguration.configuration.inventoryDefaultView);
  }
  async update(inventoryViewId, attributes, query) {
    var _query$sourceId3;
    this.logger.debug(`Trying to update inventory view with id "${inventoryViewId}"...`);
    const viewId = inventoryViewId !== null && inventoryViewId !== void 0 ? inventoryViewId : _server.SavedObjectsUtils.generateId();

    // Validate there is not a view with the same name
    await this.assertNameConflict(attributes.name, [viewId]);
    const sourceId = (_query$sourceId3 = query.sourceId) !== null && _query$sourceId3 !== void 0 ? _query$sourceId3 : InventoryViewsClient.DEFAULT_SOURCE_ID;
    const [sourceConfiguration, inventoryViewSavedObject] = await Promise.all([this.infraSources.getSourceConfiguration(this.savedObjectsClient, sourceId), this.savedObjectsClient.create(_inventory_view.inventoryViewSavedObjectName, attributes, {
      id: viewId,
      overwrite: true
    })]);
    return this.mapSavedObjectToInventoryView(inventoryViewSavedObject, sourceConfiguration.configuration.inventoryDefaultView);
  }
  delete(inventoryViewId) {
    this.logger.debug(`Trying to delete inventory view with id ${inventoryViewId} ...`);
    return this.savedObjectsClient.delete(_inventory_view.inventoryViewSavedObjectName, inventoryViewId);
  }
  mapSavedObjectToInventoryView(savedObject, defaultViewId) {
    const inventoryViewSavedObject = (0, _runtime_types.decodeOrThrow)(_types.inventoryViewSavedObjectRT)(savedObject);
    return {
      id: inventoryViewSavedObject.id,
      version: inventoryViewSavedObject.version,
      updatedAt: inventoryViewSavedObject.updated_at,
      attributes: {
        ...(0, _runtime_types.decodeOrThrow)(_inventory_views.inventoryViewAttributesRT)(inventoryViewSavedObject.attributes),
        isDefault: inventoryViewSavedObject.id === defaultViewId,
        isStatic: false
      }
    };
  }
  getAllViews() {
    return this.savedObjectsClient.find({
      type: _inventory_view.inventoryViewSavedObjectName,
      perPage: 1000 // Fetch 1 page by default with a max of 1000 results
    });
  }

  moveDefaultViewOnTop(views) {
    const defaultViewPosition = views.findIndex(view => view.attributes.isDefault);
    if (defaultViewPosition !== -1) {
      const element = views.splice(defaultViewPosition, 1)[0];
      views.unshift(element);
    }
    return views;
  }

  /**
   * We want to control conflicting names on the views
   */
  async assertNameConflict(name, whitelist = []) {
    const results = await this.getAllViews();
    const hasConflict = [InventoryViewsClient.createStaticView(), ...results.saved_objects].some(obj => !whitelist.includes(obj.id) && obj.attributes.name === name);
    if (hasConflict) {
      throw _boom.default.conflict('A view with that name already exists.');
    }
  }
}
exports.InventoryViewsClient = InventoryViewsClient;
(0, _defineProperty2.default)(InventoryViewsClient, "STATIC_VIEW_ID", '0');
(0, _defineProperty2.default)(InventoryViewsClient, "DEFAULT_SOURCE_ID", 'default');
(0, _defineProperty2.default)(InventoryViewsClient, "createStaticView", defaultViewId => ({
  id: _inventory_views.staticInventoryViewId,
  attributes: {
    ..._inventory_views.staticInventoryViewAttributes,
    isDefault: defaultViewId === InventoryViewsClient.STATIC_VIEW_ID
  }
}));