"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAnalyticsRoutes = registerAnalyticsRoutes;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _error_codes = require("../../../common/types/error_codes");
var _add_analytics_collection = require("../../lib/analytics/add_analytics_collection");
var _analytics_events_exist = require("../../lib/analytics/analytics_events_exist");
var _create_api_key = require("../../lib/analytics/create_api_key");
var _delete_analytics_collection = require("../../lib/analytics/delete_analytics_collection");
var _fetch_analytics_collection = require("../../lib/analytics/fetch_analytics_collection");
var _create_error = require("../../utils/create_error");
var _elasticsearch_error_handler = require("../../utils/elasticsearch_error_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createIndexNotFoundError = (error, response) => {
  return (0, _create_error.createError)({
    errorCode: error.message,
    message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.addAnalyticsCollection.analyticsCollectionNotFoundErrorMessage', {
      defaultMessage: 'Analytics collection not found'
    }),
    response,
    statusCode: 404
  });
};
function registerAnalyticsRoutes({
  router,
  log,
  data,
  savedObjects
}) {
  router.get({
    path: '/internal/enterprise_search/analytics/collections',
    validate: {
      query: _configSchema.schema.object({
        query: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    try {
      const query = request.query.query && request.query.query + '*';
      const collections = await (0, _fetch_analytics_collection.fetchAnalyticsCollections)(client, query);
      return response.ok({
        body: collections
      });
    } catch (error) {
      if (error.message === _error_codes.ErrorCode.ANALYTICS_COLLECTION_NOT_FOUND) {
        return response.ok({
          body: []
        });
      }
      throw error;
    }
  }));
  router.get({
    path: '/internal/enterprise_search/analytics/collections/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    try {
      const collections = await (0, _fetch_analytics_collection.fetchAnalyticsCollections)(client, request.params.name);
      return response.ok({
        body: collections[0]
      });
    } catch (error) {
      if (error.message === _error_codes.ErrorCode.ANALYTICS_COLLECTION_NOT_FOUND) {
        return createIndexNotFoundError(error, response);
      }
      throw error;
    }
  }));
  router.post({
    path: '/internal/enterprise_search/analytics/collections/{name}/api_key',
    validate: {
      body: _configSchema.schema.object({
        keyName: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const collectionName = decodeURIComponent(request.params.name);
    const {
      keyName
    } = request.body;
    const {
      client
    } = (await context.core).elasticsearch;
    const apiKey = await (0, _create_api_key.createApiKey)(client, collectionName, keyName);
    return response.ok({
      body: {
        apiKey
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: '/internal/enterprise_search/analytics/collections',
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client: elasticsearchClient
    } = (await context.core).elasticsearch;
    const dataViewsService = await data.indexPatterns.dataViewsServiceFactory(savedObjects.getScopedClient(request), elasticsearchClient.asCurrentUser, request);
    try {
      const body = await (0, _add_analytics_collection.addAnalyticsCollection)(elasticsearchClient, dataViewsService, request.body.name);
      return response.ok({
        body
      });
    } catch (error) {
      if (error.message === _error_codes.ErrorCode.ANALYTICS_COLLECTION_ALREADY_EXISTS) {
        return (0, _create_error.createError)({
          errorCode: error.message,
          message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.addAnalyticsCollection.analyticsCollectionExistsError', {
            defaultMessage: 'Collection name already exists. Choose another name.'
          }),
          response,
          statusCode: 409
        });
      }
      throw error;
    }
  }));
  router.delete({
    path: '/internal/enterprise_search/analytics/collections/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    try {
      await (0, _delete_analytics_collection.deleteAnalyticsCollectionById)(client, request.params.name);
      return response.ok();
    } catch (error) {
      if (error.message === _error_codes.ErrorCode.ANALYTICS_COLLECTION_NOT_FOUND) {
        return createIndexNotFoundError(error, response);
      }
      throw error;
    }
  }));
  router.get({
    path: '/internal/enterprise_search/analytics/collection/{name}/events/exist',
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const eventsIndexExists = await (0, _analytics_events_exist.analyticsEventsExist)(client, request.params.name);
    const body = {
      exists: eventsIndexExists
    };
    return response.ok({
      body
    });
  }));
}