"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatHit = formatHit;
var _i18n = require("@kbn/i18n");
var _format_value = require("./format_value");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const formattedHitCache = new WeakMap();

/**
 * Returns a formatted document in form of key/value pairs of the fields name and a formatted value.
 * The value returned in each pair is an HTML string which is safe to be applied to the DOM, since
 * it's formatted using field formatters.
 * @param hit
 * @param dataView
 * @param shouldShowFieldHandler
 * @param maxEntries
 * @param fieldFormats
 */
function formatHit(hit, dataView, shouldShowFieldHandler, maxEntries, fieldFormats) {
  var _hit$raw$highlight;
  const cached = formattedHitCache.get(hit.raw);
  if (cached && cached.maxEntries === maxEntries) {
    return cached.formattedHit;
  }
  const highlights = (_hit$raw$highlight = hit.raw.highlight) !== null && _hit$raw$highlight !== void 0 ? _hit$raw$highlight : {};
  // Flatten the object using the flattenHit implementation we use across Discover for flattening documents.
  const flattened = hit.flattened;
  const highlightPairs = [];
  const sourcePairs = [];

  // Add each flattened field into the corresponding array for highlighted or other fields,
  // depending on whether the original hit had a highlight for it. That way we can later
  // put highlighted fields first in the document summary.
  Object.entries(flattened).forEach(([key, val]) => {
    // Retrieve the (display) name of the fields, if it's a mapped field on the data view
    const field = dataView.fields.getByName(key);
    const displayKey = field === null || field === void 0 ? void 0 : field.displayName;
    const pairs = highlights[key] ? highlightPairs : sourcePairs;
    // Format the raw value using the regular field formatters for that field
    const formattedValue = (0, _format_value.formatFieldValue)(val, hit.raw, fieldFormats, dataView, field);
    // If the field was a mapped field, we validate it against the fieldsToShow list, if not
    // we always include it into the result.
    if (displayKey) {
      if (shouldShowFieldHandler(key)) {
        pairs.push([displayKey, formattedValue, key]);
      }
    } else {
      pairs.push([key, formattedValue, key]);
    }
  });
  const pairs = [...highlightPairs, ...sourcePairs];
  const formatted =
  // If document has more formatted fields than configured via MAX_DOC_FIELDS_DISPLAYED we cut
  // off additional fields and instead show a summary how many more field exists.
  pairs.length <= maxEntries ? pairs : [...pairs.slice(0, maxEntries), [_i18n.i18n.translate('discover.formatHit.moreFields', {
    defaultMessage: 'and {count} more {count, plural, one {field} other {fields}}',
    values: {
      count: pairs.length - maxEntries
    }
  }), '', null]];
  formattedHitCache.set(hit.raw, {
    formattedHit: formatted,
    maxEntries
  });
  return formatted;
}