"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enhancedEsSearchStrategyProvider = void 0;
var _operators = require("rxjs/operators");
var _rxjs = require("rxjs");
var _server = require("@kbn/kibana-utils-plugin/server");
var _report_search_error = require("../../report_search_error");
var _common = require("../../../../common");
var _request_utils = require("./request_utils");
var _response_utils = require("./response_utils");
var _search = require("../../collectors/search");
var _es_search = require("../es_search");
var _sanitize_request_params = require("../../sanitize_request_params");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const enhancedEsSearchStrategyProvider = (legacyConfig$, searchConfig, logger, usage, useInternalUser = false) => {
  function cancelAsyncSearch(id, esClient) {
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    return client.asyncSearch.delete({
      id
    });
  }
  function asyncSearch({
    id,
    ...request
  }, options, {
    esClient,
    uiSettingsClient
  }) {
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    const search = async () => {
      var _request$params, _request$params2, _meta$request;
      const params = id ? {
        ...(0, _request_utils.getDefaultAsyncGetParams)(searchConfig, options),
        ...((_request$params = request.params) !== null && _request$params !== void 0 && _request$params.keep_alive ? {
          keep_alive: request.params.keep_alive
        } : {}),
        ...((_request$params2 = request.params) !== null && _request$params2 !== void 0 && _request$params2.wait_for_completion_timeout ? {
          wait_for_completion_timeout: request.params.wait_for_completion_timeout
        } : {})
      } : {
        ...(await (0, _request_utils.getDefaultAsyncSubmitParams)(uiSettingsClient, searchConfig, options)),
        ...request.params
      };
      const {
        body,
        headers,
        meta
      } = id ? await client.asyncSearch.get({
        ...params,
        id
      }, {
        ...options.transport,
        signal: options.abortSignal,
        meta: true
      }) : await client.asyncSearch.submit(params, {
        ...options.transport,
        signal: options.abortSignal,
        meta: true
      });
      const response = (0, _es_search.shimHitsTotal)(body.response, options);
      return (0, _response_utils.toAsyncKibanaSearchResponse)({
        ...body,
        response
      }, headers === null || headers === void 0 ? void 0 : headers.warning,
      // do not return requestParams on polling calls
      id ? undefined : meta === null || meta === void 0 ? void 0 : (_meta$request = meta.request) === null || _meta$request === void 0 ? void 0 : _meta$request.params);
    };
    const cancel = async () => {
      if (!id || options.isStored) return;
      try {
        await cancelAsyncSearch(id, esClient);
      } catch (e) {
        var _e$meta;
        // A 404 means either this search request does not exist, or that it is already cancelled
        if (((_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.statusCode) === 404) return;

        // Log all other (unexpected) error messages
        logger.error(`cancelAsyncSearch error: ${e.message}`);
      }
    };
    return (0, _common.pollSearch)(search, cancel, {
      pollInterval: searchConfig.asyncSearch.pollInterval,
      ...options
    }).pipe((0, _operators.tap)(response => id = response.id), (0, _operators.tap)((0, _search.searchUsageObserver)(logger, usage)), (0, _operators.catchError)(e => {
      throw (0, _report_search_error.getKbnSearchError)(e);
    }));
  }
  async function rollupSearch(request, options, {
    esClient,
    uiSettingsClient
  }) {
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    const legacyConfig = await (0, _rxjs.firstValueFrom)(legacyConfig$);
    const {
      body,
      index,
      ...params
    } = request.params;
    const method = 'POST';
    const path = encodeURI(`/${index}/_rollup_search`);
    const querystring = {
      ...(0, _es_search.getShardTimeout)(legacyConfig),
      ...(await (0, _request_utils.getIgnoreThrottled)(uiSettingsClient)),
      ...(await (0, _es_search.getDefaultSearchParams)(uiSettingsClient)),
      ...params
    };
    try {
      var _esResponse$meta, _esResponse$meta$requ, _esResponse$meta2, _esResponse$meta2$req;
      const esResponse = await client.transport.request({
        method,
        path,
        body,
        querystring
      }, {
        signal: options === null || options === void 0 ? void 0 : options.abortSignal,
        meta: true
      });
      const response = esResponse.body;
      return {
        rawResponse: (0, _es_search.shimHitsTotal)(response, options),
        ...((_esResponse$meta = esResponse.meta) !== null && _esResponse$meta !== void 0 && (_esResponse$meta$requ = _esResponse$meta.request) !== null && _esResponse$meta$requ !== void 0 && _esResponse$meta$requ.params ? {
          requestParams: (0, _sanitize_request_params.sanitizeRequestParams)((_esResponse$meta2 = esResponse.meta) === null || _esResponse$meta2 === void 0 ? void 0 : (_esResponse$meta2$req = _esResponse$meta2.request) === null || _esResponse$meta2$req === void 0 ? void 0 : _esResponse$meta2$req.params)
        } : {}),
        ...(0, _es_search.getTotalLoaded)(response)
      };
    } catch (e) {
      throw (0, _report_search_error.getKbnSearchError)(e);
    }
  }
  return {
    /**
     * @param request
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Observable<IEsSearchResponse<any>>`
     * @throws `KbnSearchError`
     */
    search: (request, options, deps) => {
      logger.debug(`search ${JSON.stringify(request.params) || request.id}`);
      if (request.indexType && request.indexType !== 'rollup') {
        throw new _report_search_error.KbnSearchError('Unknown indexType', 400);
      }
      if (request.indexType === undefined || !deps.rollupsEnabled) {
        return asyncSearch(request, options, deps);
      } else {
        return (0, _rxjs.from)(rollupSearch(request, options, deps));
      }
    },
    /**
     * @param id async search ID to cancel, as returned from _async_search API
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Promise<void>`
     * @throws `KbnServerError`
     */
    cancel: async (id, options, {
      esClient
    }) => {
      logger.debug(`cancel ${id}`);
      try {
        await cancelAsyncSearch(id, esClient);
      } catch (e) {
        throw (0, _server.getKbnServerError)(e);
      }
    },
    /**
     *
     * @param id async search ID to extend, as returned from _async_search API
     * @param keepAlive
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Promise<void>`
     * @throws `KbnServerError`
     */
    extend: async (id, keepAlive, options, {
      esClient
    }) => {
      logger.debug(`extend ${id} by ${keepAlive}`);
      try {
        const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
        await client.asyncSearch.get({
          id,
          keep_alive: keepAlive
        });
      } catch (e) {
        throw (0, _server.getKbnServerError)(e);
      }
    }
  };
};
exports.enhancedEsSearchStrategyProvider = enhancedEsSearchStrategyProvider;