"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEsqlFn = void 0;
var _fieldTypes = require("@kbn/field-types");
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/inspector-plugin/common");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _esQuery = require("@kbn/es-query");
var _es_query = require("../../es_query");
var _query = require("../../query");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function normalizeType(type) {
  switch (type) {
    case _fieldTypes.ES_FIELD_TYPES._INDEX:
    case _fieldTypes.ES_FIELD_TYPES.GEO_POINT:
    case _fieldTypes.ES_FIELD_TYPES.IP:
      return _fieldTypes.KBN_FIELD_TYPES.STRING;
    case '_version':
      return _fieldTypes.KBN_FIELD_TYPES.NUMBER;
    case 'datetime':
      return _fieldTypes.KBN_FIELD_TYPES.DATE;
    default:
      return (0, _fieldTypes.castEsToKbnFieldTypeName)(type);
  }
}
function extractTypeAndReason(attributes) {
  if (['type', 'reason'].every(prop => prop in attributes)) {
    return attributes;
  }
  if ('error' in attributes) {
    return extractTypeAndReason(attributes.error);
  }
  return {};
}
const getEsqlFn = ({
  getStartDependencies
}) => {
  const essql = {
    name: 'esql',
    type: 'datatable',
    inputTypes: ['kibana_context', 'null'],
    help: _i18n.i18n.translate('data.search.esql.help', {
      defaultMessage: 'Queries Elasticsearch using ES|QL.'
    }),
    args: {
      query: {
        aliases: ['_', 'q'],
        types: ['string'],
        help: _i18n.i18n.translate('data.search.esql.query.help', {
          defaultMessage: 'An ES|QL query.'
        })
      },
      // timezone: {
      //   aliases: ['tz'],
      //   types: ['string'],
      //   default: 'UTC',
      //   help: i18n.translate('data.search.esql.timezone.help', {
      //     defaultMessage:
      //       'The timezone to use for date operations. Valid ISO8601 formats and UTC offsets both work.',
      //   }),
      // },
      timeField: {
        aliases: ['timeField'],
        types: ['string'],
        help: _i18n.i18n.translate('data.search.essql.timeField.help', {
          defaultMessage: 'The time field to use in the time range filter set in the context.'
        })
      },
      locale: {
        aliases: ['locale'],
        types: ['string'],
        help: _i18n.i18n.translate('data.search.essql.locale.help', {
          defaultMessage: 'The locale to use.'
        })
      }
    },
    fn(input, {
      query,
      /* timezone, */timeField,
      locale
    }, {
      abortSignal,
      inspectorAdapters,
      getKibanaRequest
    }) {
      return (0, _rxjs.defer)(() => getStartDependencies(() => {
        const request = getKibanaRequest === null || getKibanaRequest === void 0 ? void 0 : getKibanaRequest();
        if (!request) {
          throw new Error('A KibanaRequest is required to run queries on the server. ' + 'Please provide a request object to the expression execution params.');
        }
        return request;
      })).pipe((0, _operators.switchMap)(({
        search,
        uiSettings
      }) => {
        const params = {
          query,
          // time_zone: timezone,
          locale
        };
        if (input) {
          var _input$filters;
          const esQueryConfigs = (0, _es_query.getEsQueryConfig)(uiSettings);
          const timeFilter = input.timeRange && (0, _query.getTime)(undefined, input.timeRange, {
            fieldName: timeField
          });
          params.filter = (0, _esQuery.buildEsQuery)(undefined, input.query || [], [...((_input$filters = input.filters) !== null && _input$filters !== void 0 ? _input$filters : []), ...(timeFilter ? [timeFilter] : [])], esQueryConfigs);
        }
        let startTime = Date.now();
        const logInspectorRequest = () => {
          if (!inspectorAdapters.requests) {
            inspectorAdapters.requests = new _common.RequestAdapter();
          }
          const request = inspectorAdapters.requests.start(_i18n.i18n.translate('data.search.dataRequest.title', {
            defaultMessage: 'Data'
          }), {
            description: _i18n.i18n.translate('data.search.es_search.dataRequest.description', {
              defaultMessage: 'This request queries Elasticsearch to fetch the data for the visualization.'
            })
          }, startTime);
          startTime = Date.now();
          return request;
        };
        return search({
          params
        }, {
          abortSignal,
          strategy: _.ESQL_SEARCH_STRATEGY
        }).pipe((0, _operators.catchError)(error => {
          if (!error.err) {
            error.message = `Unexpected error from Elasticsearch: ${error.message}`;
          } else {
            const {
              type,
              reason
            } = extractTypeAndReason(error.err.attributes);
            if (type === 'parsing_exception') {
              error.message = `Couldn't parse Elasticsearch ES|QL query. Check your query and try again. Error: ${reason}`;
            } else {
              error.message = `Unexpected error from Elasticsearch: ${type} - ${reason}`;
            }
          }
          return (0, _rxjs.throwError)(() => error);
        }), (0, _operators.tap)({
          next({
            rawResponse,
            requestParams
          }) {
            logInspectorRequest().stats({
              hits: {
                label: _i18n.i18n.translate('data.search.es_search.hitsLabel', {
                  defaultMessage: 'Hits'
                }),
                value: `${rawResponse.values.length}`,
                description: _i18n.i18n.translate('data.search.es_search.hitsDescription', {
                  defaultMessage: 'The number of documents returned by the query.'
                })
              }
            }).json(params).ok({
              json: rawResponse,
              requestParams
            });
          },
          error(error) {
            logInspectorRequest().json(params).error({
              json: 'attributes' in error ? error.attributes : {
                message: error.message
              }
            });
          }
        }));
      }), (0, _operators.map)(({
        rawResponse: body,
        warning
      }) => {
        var _body$columns$map, _body$columns;
        const columns = (_body$columns$map = (_body$columns = body.columns) === null || _body$columns === void 0 ? void 0 : _body$columns.map(({
          name,
          type
        }) => ({
          id: name,
          name,
          meta: {
            type: normalizeType(type)
          }
        }))) !== null && _body$columns$map !== void 0 ? _body$columns$map : [];
        const columnNames = columns.map(({
          name
        }) => name);
        const rows = body.values.map(row => (0, _lodash.zipObject)(columnNames, row));
        return {
          type: 'datatable',
          meta: {
            type: 'es_ql'
          },
          columns,
          rows,
          warning
        };
      }));
    }
  };
  return essql;
};
exports.getEsqlFn = getEsqlFn;