"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.unfollowLeaderIndex = exports.selectEditFollowerIndex = exports.selectDetailFollowerIndex = exports.saveFollowerIndex = exports.resumeFollowerIndex = exports.pauseFollowerIndex = exports.loadFollowerIndices = exports.getFollowerIndex = void 0;
var _i18n = require("@kbn/i18n");
var _routing = require("../../services/routing");
var _notifications = require("../../services/notifications");
var _constants = require("../../constants");
var _api = require("../../services/api");
var t = _interopRequireWildcard(require("../action_types"));
var _api2 = require("./api");
var _selectors = require("../selectors");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const {
  FOLLOWER_INDEX: scope
} = _constants.SECTIONS;
const selectDetailFollowerIndex = id => ({
  type: t.FOLLOWER_INDEX_SELECT_DETAIL,
  payload: id
});
exports.selectDetailFollowerIndex = selectDetailFollowerIndex;
const selectEditFollowerIndex = id => ({
  type: t.FOLLOWER_INDEX_SELECT_EDIT,
  payload: id
});
exports.selectEditFollowerIndex = selectEditFollowerIndex;
const loadFollowerIndices = (isUpdating = false) => (0, _api2.sendApiRequest)({
  label: t.FOLLOWER_INDEX_LOAD,
  scope,
  status: isUpdating ? _constants.API_STATUS.UPDATING : _constants.API_STATUS.LOADING,
  handler: async () => await (0, _api.loadFollowerIndices)(isUpdating)
});
exports.loadFollowerIndices = loadFollowerIndices;
const getFollowerIndex = id => (0, _api2.sendApiRequest)({
  label: t.FOLLOWER_INDEX_GET,
  scope: `${scope}-get`,
  handler: async () => await (0, _api.getFollowerIndex)(id)
});
exports.getFollowerIndex = getFollowerIndex;
const saveFollowerIndex = (name, followerIndex, isUpdating = false) => (0, _api2.sendApiRequest)({
  label: t.FOLLOWER_INDEX_CREATE,
  status: _constants.API_STATUS.SAVING,
  scope: `${scope}-save`,
  handler: async () => {
    if (isUpdating) {
      return await (0, _api.updateFollowerIndex)(name, followerIndex);
    }
    return await (0, _api.createFollowerIndex)({
      name,
      ...followerIndex
    });
  },
  onSuccess() {
    const successMessage = isUpdating ? _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.updateAction.successNotificationTitle', {
      defaultMessage: `Follower index '{name}' updated successfully`,
      values: {
        name
      }
    }) : _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.addAction.successNotificationTitle', {
      defaultMessage: `Added follower index '{name}'`,
      values: {
        name
      }
    });
    (0, _notifications.getToasts)().addSuccess(successMessage);
    _routing.routing.navigate(`/follower_indices`, {
      name: encodeURIComponent(name)
    });
  }
});
exports.saveFollowerIndex = saveFollowerIndex;
const pauseFollowerIndex = id => (0, _api2.sendApiRequest)({
  label: t.FOLLOWER_INDEX_PAUSE,
  status: _constants.API_STATUS.SAVING,
  scope,
  handler: async () => (0, _api.pauseFollowerIndex)(id),
  onSuccess(response, dispatch) {
    /**
     * We can have 1 or more follower index pause operation
     * that can fail or succeed. We will show 1 toast notification for each.
     */
    if (response.errors.length) {
      const hasMultipleErrors = response.errors.length > 1;
      const errorMessage = hasMultipleErrors ? _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.pauseAction.errorMultipleNotificationTitle', {
        defaultMessage: `Error pausing {count} follower indices`,
        values: {
          count: response.errors.length
        }
      }) : _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.pauseAction.errorSingleNotificationTitle', {
        defaultMessage: `Error pausing follower index '{name}'`,
        values: {
          name: response.errors[0].id
        }
      });
      (0, _notifications.getToasts)().addDanger(errorMessage);
    }
    if (response.itemsPaused.length) {
      const hasMultiplePaused = response.itemsPaused.length > 1;
      const successMessage = hasMultiplePaused ? _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.pauseAction.successMultipleNotificationTitle', {
        defaultMessage: `{count} follower indices were paused`,
        values: {
          count: response.itemsPaused.length
        }
      }) : _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.pauseAction.successSingleNotificationTitle', {
        defaultMessage: `Follower index '{name}' was paused`,
        values: {
          name: response.itemsPaused[0]
        }
      });
      (0, _notifications.getToasts)().addSuccess(successMessage);

      // Refresh list
      dispatch(loadFollowerIndices(true));
    }
  }
});
exports.pauseFollowerIndex = pauseFollowerIndex;
const resumeFollowerIndex = id => (0, _api2.sendApiRequest)({
  label: t.FOLLOWER_INDEX_RESUME,
  status: _constants.API_STATUS.SAVING,
  scope,
  handler: async () => (0, _api.resumeFollowerIndex)(id),
  onSuccess(response, dispatch) {
    /**
     * We can have 1 or more follower index resume operation
     * that can fail or succeed. We will show 1 toast notification for each.
     */
    if (response.errors.length) {
      const hasMultipleErrors = response.errors.length > 1;
      const errorMessage = hasMultipleErrors ? _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.resumeAction.errorMultipleNotificationTitle', {
        defaultMessage: `Error resuming {count} follower indices`,
        values: {
          count: response.errors.length
        }
      }) : _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.resumeAction.errorSingleNotificationTitle', {
        defaultMessage: `Error resuming follower index '{name}'`,
        values: {
          name: response.errors[0].id
        }
      });
      (0, _notifications.getToasts)().addDanger(errorMessage);
    }
    if (response.itemsResumed.length) {
      const hasMultipleResumed = response.itemsResumed.length > 1;
      const successMessage = hasMultipleResumed ? _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.resumeAction.successMultipleNotificationTitle', {
        defaultMessage: `{count} follower indices were resumed`,
        values: {
          count: response.itemsResumed.length
        }
      }) : _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.resumeAction.successSingleNotificationTitle', {
        defaultMessage: `Follower index '{name}' was resumed`,
        values: {
          name: response.itemsResumed[0]
        }
      });
      (0, _notifications.getToasts)().addSuccess(successMessage);
    }

    // Refresh list
    dispatch(loadFollowerIndices(true));
  }
});
exports.resumeFollowerIndex = resumeFollowerIndex;
const unfollowLeaderIndex = id => (0, _api2.sendApiRequest)({
  label: t.FOLLOWER_INDEX_UNFOLLOW,
  status: _constants.API_STATUS.DELETING,
  scope: `${scope}-delete`,
  handler: async () => (0, _api.unfollowLeaderIndex)(id),
  onSuccess(response, dispatch, getState) {
    /**
     * We can have 1 or more follower index unfollow operation
     * that can fail or succeed. We will show 1 toast notification for each.
     */
    if (response.errors.length) {
      const hasMultipleErrors = response.errors.length > 1;
      const errorMessage = hasMultipleErrors ? _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.unfollowAction.errorMultipleNotificationTitle', {
        defaultMessage: `Error unfollowing leader index of {count} follower indices`,
        values: {
          count: response.errors.length
        }
      }) : _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.unfollowAction.errorSingleNotificationTitle', {
        defaultMessage: `Error unfollowing leader index of follower index '{name}'`,
        values: {
          name: response.errors[0].id
        }
      });
      (0, _notifications.getToasts)().addDanger(errorMessage);
    }
    if (response.itemsUnfollowed.length) {
      const hasMultipleUnfollow = response.itemsUnfollowed.length > 1;
      const successMessage = hasMultipleUnfollow ? _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.unfollowAction.successMultipleNotificationTitle', {
        defaultMessage: `Leader indices of {count} follower indices were unfollowed`,
        values: {
          count: response.itemsUnfollowed.length
        }
      }) : _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.unfollowAction.successSingleNotificationTitle', {
        defaultMessage: `Leader index of follower index '{name}' was unfollowed`,
        values: {
          name: response.itemsUnfollowed[0]
        }
      });
      (0, _notifications.getToasts)().addSuccess(successMessage);
    }
    if (response.itemsNotOpen.length) {
      const hasMultipleNotOpen = response.itemsNotOpen.length > 1;
      const warningMessage = hasMultipleNotOpen ? _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.unfollowAction.notOpenWarningMultipleNotificationTitle', {
        defaultMessage: `{count} indices could not be re-opened`,
        values: {
          count: response.itemsNotOpen.length
        }
      }) : _i18n.i18n.translate('xpack.crossClusterReplication.followerIndex.unfollowAction.notOpenWarningSingleNotificationTitle', {
        defaultMessage: `Index '{name}' could not be re-opened`,
        values: {
          name: response.itemsNotOpen[0]
        }
      });
      (0, _notifications.getToasts)().addWarning(warningMessage);
    }

    // If we've just unfollowed a follower index we were looking at, we need to close the panel.
    const followerIndexId = (0, _selectors.getSelectedFollowerIndexId)('detail')(getState());
    if (response.itemsUnfollowed.includes(followerIndexId)) {
      dispatch(selectDetailFollowerIndex(null));
    }
  }
});
exports.unfollowLeaderIndex = unfollowLeaderIndex;