"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performRemoveReferencesTo = void 0;
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _search = require("../search");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const performRemoveReferencesTo = async ({
  type,
  id,
  options
}, {
  registry,
  helpers,
  client,
  mappings,
  extensions = {}
}) => {
  var _body$failures;
  const {
    common: commonHelper
  } = helpers;
  const {
    securityExtension
  } = extensions;
  const namespace = commonHelper.getCurrentNamespace(options.namespace);
  const {
    refresh = true
  } = options;
  await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.authorizeRemoveReferences({
    namespace,
    object: {
      type,
      id
    }
  }));
  const allTypes = registry.getAllTypes().map(t => t.name);

  // we need to target all SO indices as all types of objects may have references to the given SO.
  const targetIndices = commonHelper.getIndicesForTypes(allTypes);
  const {
    body,
    statusCode,
    headers
  } = await client.updateByQuery({
    index: targetIndices,
    refresh,
    body: {
      script: {
        source: `
              if (ctx._source.containsKey('references')) {
                def items_to_remove = [];
                for (item in ctx._source.references) {
                  if ( (item['type'] == params['type']) && (item['id'] == params['id']) ) {
                    items_to_remove.add(item);
                  }
                }
                ctx._source.references.removeAll(items_to_remove);
              }
            `,
        params: {
          type,
          id
        },
        lang: 'painless'
      },
      conflicts: 'proceed',
      ...(0, _search.getSearchDsl)(mappings, registry, {
        namespaces: namespace ? [namespace] : undefined,
        type: allTypes,
        hasReference: {
          type,
          id
        }
      })
    }
  }, {
    ignore: [404],
    meta: true
  });
  // fail fast if we can't verify a 404 is from Elasticsearch
  if ((0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
    statusCode,
    headers
  })) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError(type, id);
  }
  if ((_body$failures = body.failures) !== null && _body$failures !== void 0 && _body$failures.length) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id, `${body.failures.length} references could not be removed`);
  }
  return {
    updated: body.updated
  };
};
exports.performRemoveReferencesTo = performRemoveReferencesTo;