"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.gatherV2CgroupMetrics = gatherV2CgroupMetrics;
var _promises = _interopRequireDefault(require("fs/promises"));
var _path = require("path");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const PROC_CGROUP2_DIR = '/sys/fs/cgroup';
const CPU_STATS_FILE = 'cpu.stat';
const CPU_MAX_FILE = 'cpu.max';
const MEMORY_CURRENT_FILE = 'memory.current';
const MEMORY_SWAP_CURRENT_FILE = 'memory.swap.current';
const getCGroupFilePath = (group, fileName) => {
  return (0, _path.join)(PROC_CGROUP2_DIR, group, fileName);
};
async function gatherV2CgroupMetrics(group) {
  const [{
    usage_nanos: usageNanos,
    ...stat
  }, cpuMax, memoryCurrent, swapCurrent] = await Promise.all([readCPUStat(group), readCPUMax(group), readMemoryCurrent(group), readSwapCurrent(group)]);
  return {
    cpu: {
      ...cpuMax,
      control_group: group,
      stat
    },
    cpuacct: {
      control_group: group,
      usage_nanos: usageNanos
    },
    cgroup_memory: {
      current_in_bytes: memoryCurrent,
      swap_current_in_bytes: swapCurrent
    }
  };
}
async function readMemoryCurrent(group) {
  const rawMemoryCurrent = (await _promises.default.readFile(getCGroupFilePath(group, MEMORY_CURRENT_FILE))).toString().trim();
  return parseInt(rawMemoryCurrent, 10);
}
async function readSwapCurrent(group) {
  const rawMemoryCurrent = (await _promises.default.readFile(getCGroupFilePath(group, MEMORY_SWAP_CURRENT_FILE))).toString().trim();
  return parseInt(rawMemoryCurrent, 10);
}
async function readCPUMax(group) {
  const [quota, period] = (await _promises.default.readFile(getCGroupFilePath(group, CPU_MAX_FILE))).toString().trim().split(/\s+/);
  return {
    cfs_quota_micros: quota === 'max' ? -1 : parseInt(quota, 10),
    cfs_period_micros: parseInt(period, 10)
  };
}
async function readCPUStat(group) {
  const stat = {
    number_of_elapsed_periods: -1,
    number_of_times_throttled: -1,
    time_throttled_nanos: -1,
    usage_nanos: -1
  };
  return (await _promises.default.readFile(getCGroupFilePath(group, CPU_STATS_FILE))).toString().split(/\n/).reduce((acc, line) => {
    const [key, value] = line.split(/\s+/);
    switch (key) {
      case 'nr_periods':
        acc.number_of_elapsed_periods = parseInt(value, 10);
        break;
      case 'nr_throttled':
        acc.number_of_times_throttled = parseInt(value, 10);
        break;
      case 'throttled_usec':
        acc.time_throttled_nanos = parseInt(value, 10);
        break;
      // In V2 cpuacct also lives in cpu.stat
      case 'usage_usec':
        acc.usage_nanos = parseInt(value, 10);
        break;
    }
    return stat;
  }, stat);
}