"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _http = require("http");
var _https = require("https");
var _get_agents_sockets_stats = require("./get_agents_sockets_stats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const HTTPS = 'https:';

/**
 * Exposes the APIs to fetch stats of the existing agents.
 */

/**
 * Allows obtaining Agent factories, which can then be fed into elasticsearch-js's Client class.
 * Ideally, we should obtain one Agent factory for each ES Client class.
 * This allows using the same Agent across all the Pools and Connections of the Client (one per ES node).
 *
 * Agent instances are stored internally to allow collecting metrics (nbr of active/idle connections to ES).
 *
 * Using the same Agent factory across multiple ES Client instances is strongly discouraged, cause ES Client
 * exposes methods that can modify the underlying pools, effectively impacting the connections of other Clients.
 * @internal
 **/
class AgentManager {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "agents", void 0);
    this.logger = logger;
    this.agents = new Set();
  }
  getAgentFactory(agentOptions) {
    // a given agent factory always provides the same Agent instances (for the same protocol)
    // we keep references to the instances at factory level, to be able to reuse them
    let httpAgent;
    let httpsAgent;
    return connectionOpts => {
      if (connectionOpts.url.protocol === HTTPS) {
        if (!httpsAgent) {
          const config = Object.assign({}, agentOptions, connectionOpts.tls);
          httpsAgent = new _https.Agent(config);
          this.agents.add(httpsAgent);
          dereferenceOnDestroy(this.agents, httpsAgent);
        }
        return httpsAgent;
      }
      if (!httpAgent) {
        httpAgent = new _http.Agent(agentOptions);
        this.agents.add(httpAgent);
        dereferenceOnDestroy(this.agents, httpAgent);
      }
      return httpAgent;
    };
  }
  getAgentsStats() {
    const stats = (0, _get_agents_sockets_stats.getAgentsSocketsStats)(this.agents);
    if (stats.totalQueuedRequests > 0) {
      this.logger.warn(`There are ${stats.totalQueuedRequests} queued requests. If this number is constantly high, consider scaling Kibana horizontally or increasing "elasticsearch.maxSockets" in the config.`);
    }
    return stats;
  }
}
exports.AgentManager = AgentManager;
const dereferenceOnDestroy = (store, agent) => {
  const doDestroy = agent.destroy.bind(agent);
  agent.destroy = () => {
    store.delete(agent);
    doDestroy();
  };
};