"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCapabilitiesResolver = void 0;
var _lodash = require("lodash");
var _apmUtils = require("@kbn/apm-utils");
var _resolve_helpers = require("./resolve_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const getCapabilitiesResolver = (getCapabilities, getSwitchers) => {
  let initialized = false;
  let capabilities;
  let switchers;
  // memoize is on the first argument only by default, which is what we want here
  const getSwitcherForPath = (0, _lodash.memoize)(getSwitchersToUseForPath);
  let getAggregatedSwitchers;
  return async ({
    request,
    capabilityPath,
    applications,
    useDefaultCapabilities
  }) => {
    // initialize on first call (can't do it before as we need to wait plugins start to complete)
    if (!initialized) {
      capabilities = getCapabilities();
      switchers = new Map();
      getSwitchers().forEach(switcher => {
        const switcherId = (0, _lodash.uniqueId)('s-');
        switchers.set(switcherId, {
          id: switcherId,
          ...switcher
        });
      });
      getAggregatedSwitchers = (0, _lodash.memoize)(buildGetAggregatedSwitchers(getSwitcherForPath, switchers), capabilityPaths => capabilityPaths.join('|'));
      initialized = true;
    }
    return (0, _apmUtils.withSpan)({
      name: 'resolve capabilities',
      type: 'capabilities'
    }, () => resolveCapabilities({
      capabilities,
      request,
      capabilityPath,
      applications,
      useDefaultCapabilities,
      getAggregatedSwitchers
    }));
  };
};
exports.getCapabilitiesResolver = getCapabilitiesResolver;
const resolveCapabilities = async ({
  capabilities,
  request,
  capabilityPath,
  applications,
  useDefaultCapabilities,
  getAggregatedSwitchers
}) => {
  const mergedCaps = (0, _lodash.cloneDeep)({
    ...capabilities,
    navLinks: applications.reduce((acc, app) => {
      acc[app] = true;
      return acc;
    }, capabilities.navLinks)
  });
  const switchers = getAggregatedSwitchers(capabilityPath);
  return switchers.reduce(async (caps, switcher) => {
    const resolvedCaps = await caps;
    const changes = await switcher(request, resolvedCaps, useDefaultCapabilities);
    return recursiveApplyChanges(resolvedCaps, changes);
  }, Promise.resolve(mergedCaps));
};
function recursiveApplyChanges(destination, source) {
  return Object.keys(destination).map(key => {
    const orig = destination[key];
    const changed = source[key];
    if (changed == null) {
      return [key, orig];
    }
    if (typeof orig === 'object' && typeof changed === 'object') {
      return [key, recursiveApplyChanges(orig, changed)];
    }
    return [key, typeof orig === typeof changed ? changed : orig];
  }).reduce((acc, [key, value]) => {
    acc[key] = value;
    return acc;
  }, {});
}
const getSwitchersToUseForPath = (path, switchers) => {
  const switcherIds = [];
  switchers.forEach(switcher => {
    if (switcher.capabilityPath.some(switcherPath => (0, _resolve_helpers.pathsIntersect)(path, switcherPath))) {
      switcherIds.push(switcher.id);
    }
  });
  return switcherIds;
};
const buildGetAggregatedSwitchers = (getSwitcherForPath, switcherMap) => capabilityPaths => {
  // find switchers that should be applied for the provided capabilityPaths
  const allSwitchers = [...switcherMap.values()];
  const switcherIdsToApply = new Set();
  capabilityPaths.forEach(path => {
    getSwitcherForPath(path, allSwitchers).forEach(switcherId => switcherIdsToApply.add(switcherId));
  });
  const switchersToApply = [...switcherIdsToApply].map(switcherId => switcherMap.get(switcherId));

  // split the switchers into buckets for parallel execution
  const switcherBuckets = (0, _resolve_helpers.splitIntoBuckets)(switchersToApply);

  // convert the multi-switcher buckets into switchers
  return switcherBuckets.map(_resolve_helpers.convertBucketToSwitcher);
};