"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCheapSuggestionAggregationBuilder = void 0;
var _lodash = require("lodash");
var _common = require("@kbn/data-views-plugin/common");
var _ip_search = require("../../common/options_list/ip_search");
var _options_list_suggestion_query_helpers = require("./options_list_suggestion_query_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Suggestion aggregations
 */
const getCheapSuggestionAggregationBuilder = ({
  fieldSpec
}) => {
  if ((fieldSpec === null || fieldSpec === void 0 ? void 0 : fieldSpec.type) === 'boolean') {
    return cheapSuggestionAggSubtypes.boolean;
  }
  if ((fieldSpec === null || fieldSpec === void 0 ? void 0 : fieldSpec.type) === 'ip') {
    return cheapSuggestionAggSubtypes.ip;
  }
  if (fieldSpec && (0, _common.getFieldSubtypeNested)(fieldSpec)) {
    return cheapSuggestionAggSubtypes.subtypeNested;
  }
  return cheapSuggestionAggSubtypes.keywordOrText;
};
exports.getCheapSuggestionAggregationBuilder = getCheapSuggestionAggregationBuilder;
const cheapSuggestionAggSubtypes = {
  /**
   * The "textOrKeyword" query / parser should be used whenever the field is built on some type non-nested string field
   * (such as a keyword field or a keyword+text multi-field)
   */
  keywordOrText: {
    buildAggregation: ({
      fieldName,
      fieldSpec,
      searchString,
      sort
    }) => ({
      suggestions: {
        terms: {
          field: fieldName,
          // disabling for date fields because applying a search string will return an error
          ...((fieldSpec === null || fieldSpec === void 0 ? void 0 : fieldSpec.type) !== 'date' && searchString && searchString.length > 0 ? {
            include: `${(0, _options_list_suggestion_query_helpers.getEscapedRegexQuery)(searchString)}.*`
          } : {}),
          shard_size: 10,
          order: (0, _options_list_suggestion_query_helpers.getSortType)(sort)
        }
      }
    }),
    parse: rawEsResult => {
      var _get;
      return {
        suggestions: (_get = (0, _lodash.get)(rawEsResult, 'aggregations.suggestions.buckets')) === null || _get === void 0 ? void 0 : _get.reduce((acc, suggestion) => {
          acc.push({
            value: suggestion.key,
            docCount: suggestion.doc_count
          });
          return acc;
        }, [])
      };
    }
  },
  /**
   * the "Boolean" query / parser should be used when the options list is built on a field of type boolean. The query is slightly different than a keyword query.
   */
  boolean: {
    buildAggregation: ({
      fieldName,
      sort
    }) => ({
      suggestions: {
        terms: {
          field: fieldName,
          shard_size: 10,
          order: (0, _options_list_suggestion_query_helpers.getSortType)(sort)
        }
      }
    }),
    parse: rawEsResult => {
      var _get2;
      return {
        suggestions: (_get2 = (0, _lodash.get)(rawEsResult, 'aggregations.suggestions.buckets')) === null || _get2 === void 0 ? void 0 : _get2.reduce((acc, suggestion) => {
          acc.push({
            value: suggestion.key_as_string,
            docCount: suggestion.doc_count
          });
          return acc;
        }, [])
      };
    }
  },
  /**
   * the "IP" query / parser should be used when the options list is built on a field of type IP.
   */
  ip: {
    buildAggregation: ({
      fieldName,
      searchString,
      sort
    }) => {
      let ipRangeQuery = {
        validSearch: true,
        rangeQuery: [{
          key: 'ipv6',
          from: '::',
          to: 'ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff'
        }]
      };
      if (searchString && searchString.length > 0) {
        ipRangeQuery = (0, _ip_search.getIpRangeQuery)(searchString);
        if (!ipRangeQuery.validSearch) {
          // ideally should be prevented on the client side but, if somehow an invalid search gets through to the server,
          // simply don't return an aggregation query for the ES search request
          return undefined;
        }
      }
      return {
        suggestions: {
          ip_range: {
            field: fieldName,
            ranges: ipRangeQuery.rangeQuery,
            keyed: true
          },
          aggs: {
            filteredSuggestions: {
              terms: {
                field: fieldName,
                shard_size: 10,
                order: (0, _options_list_suggestion_query_helpers.getSortType)(sort)
              }
            }
          }
        }
      };
    },
    parse: (rawEsResult, {
      sort
    }) => {
      var _rawEsResult$aggregat;
      if (!Boolean((_rawEsResult$aggregat = rawEsResult.aggregations) === null || _rawEsResult$aggregat === void 0 ? void 0 : _rawEsResult$aggregat.suggestions)) {
        // if this is happens, that means there is an invalid search that snuck through to the server side code;
        // so, might as well early return with no suggestions
        return {
          suggestions: []
        };
      }
      const buckets = [];
      (0, _options_list_suggestion_query_helpers.getIpBuckets)(rawEsResult, buckets, 'ipv4'); // modifies buckets array directly, i.e. "by reference"
      (0, _options_list_suggestion_query_helpers.getIpBuckets)(rawEsResult, buckets, 'ipv6');
      const sortedSuggestions = (sort === null || sort === void 0 ? void 0 : sort.direction) === 'asc' ? buckets.sort((bucketA, bucketB) => bucketA.doc_count - bucketB.doc_count) : buckets.sort((bucketA, bucketB) => bucketB.doc_count - bucketA.doc_count);
      return {
        suggestions: sortedSuggestions.slice(0, 10) // only return top 10 results
        .reduce((acc, suggestion) => {
          acc.push({
            value: suggestion.key,
            docCount: suggestion.doc_count
          });
          return acc;
        }, [])
      };
    }
  },
  /**
   * the "Subtype Nested" query / parser should be used when the options list is built on a field with subtype nested.
   */
  subtypeNested: {
    buildAggregation: req => {
      const {
        fieldSpec,
        fieldName,
        searchString,
        sort
      } = req;
      const subTypeNested = fieldSpec && (0, _common.getFieldSubtypeNested)(fieldSpec);
      if (!subTypeNested) {
        // if this field is not subtype nested, fall back to keywordOnly
        return cheapSuggestionAggSubtypes.keywordOnly.buildAggregation(req);
      }
      return {
        nestedSuggestions: {
          nested: {
            path: subTypeNested.nested.path
          },
          aggs: {
            suggestions: {
              terms: {
                field: fieldName,
                ...(searchString && searchString.length > 0 ? {
                  include: `${(0, _options_list_suggestion_query_helpers.getEscapedRegexQuery)(searchString)}.*`
                } : {}),
                shard_size: 10,
                order: (0, _options_list_suggestion_query_helpers.getSortType)(sort)
              }
            }
          }
        }
      };
    },
    parse: rawEsResult => {
      var _get3;
      return {
        suggestions: (_get3 = (0, _lodash.get)(rawEsResult, 'aggregations.nestedSuggestions.suggestions.buckets')) === null || _get3 === void 0 ? void 0 : _get3.reduce((acc, suggestion) => {
          acc.push({
            value: suggestion.key,
            docCount: suggestion.doc_count
          });
          return acc;
        }, [])
      };
    }
  }
};