"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTrendsQuery = exports.getTrends = exports.formatTrends = void 0;
var _helpers = require("../../../common/utils/helpers");
var _constants = require("../../../common/constants");
var _mapping_field_util = require("../../lib/mapping_field_util");
var _findings_stats_task = require("../../tasks/findings_stats_task");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getTrendsQuery = policyTemplate => ({
  index: _constants.BENCHMARK_SCORE_INDEX_DEFAULT_NS,
  // Amount of samples of the last 24 hours (accounting that we take a sample every 5 minutes)
  size: 24 * 60 / _findings_stats_task.CSPM_FINDINGS_STATS_INTERVAL,
  sort: '@timestamp:desc',
  query: {
    bool: {
      filter: [{
        term: {
          policy_template: policyTemplate
        }
      }],
      must: {
        range: {
          '@timestamp': {
            gte: 'now-1d',
            lte: 'now'
          }
        }
      }
    }
  }
});
exports.getTrendsQuery = getTrendsQuery;
const formatTrends = scoreTrendDocs => {
  return scoreTrendDocs.map(data => {
    return {
      timestamp: data['@timestamp'],
      summary: {
        totalFindings: data.total_findings,
        totalFailed: data.failed_findings,
        totalPassed: data.passed_findings,
        postureScore: (0, _helpers.calculatePostureScore)(data.passed_findings, data.failed_findings)
      },
      clusters: Object.fromEntries(Object.entries(data.score_by_cluster_id).map(([clusterId, cluster]) => [clusterId, {
        totalFindings: cluster.total_findings,
        totalFailed: cluster.failed_findings,
        totalPassed: cluster.passed_findings,
        postureScore: (0, _helpers.calculatePostureScore)(cluster.passed_findings, cluster.failed_findings)
      }])),
      benchmarks: data.score_by_benchmark_id ? Object.fromEntries(Object.entries(data.score_by_benchmark_id).flatMap(([benchmarkId, benchmark]) => Object.entries(benchmark).map(([benchmarkVersion, benchmarkStats]) => {
        const benchmarkIdVersion = (0, _mapping_field_util.toBenchmarkDocFieldKey)(benchmarkId, benchmarkVersion);
        return [benchmarkIdVersion, {
          totalFindings: benchmarkStats.total_findings,
          totalFailed: benchmarkStats.failed_findings,
          totalPassed: benchmarkStats.passed_findings,
          postureScore: (0, _helpers.calculatePostureScore)(benchmarkStats.passed_findings, benchmarkStats.failed_findings)
        }];
      }))) : {}
    };
  });
};
exports.formatTrends = formatTrends;
const getTrends = async (esClient, policyTemplate, logger) => {
  try {
    const trendsQueryResult = await esClient.search(getTrendsQuery(policyTemplate));
    if (!trendsQueryResult.hits.hits) throw new Error('missing trend results from score index');
    const scoreTrendDocs = trendsQueryResult.hits.hits.map(hit => {
      if (!hit._source) throw new Error('missing _source data for one or more of trend results');
      return hit._source;
    });
    return formatTrends(scoreTrendDocs);
  } catch (err) {
    logger.error(`Failed to fetch trendlines data ${err.message}`);
    logger.error(err);
    throw err;
  }
};
exports.getTrends = getTrends;