"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getInstalledPolicyTemplates = exports.getCspPackagePolicies = exports.getCspAgentPolicies = exports.getAgentStatusesByAgentPolicies = exports.PACKAGE_POLICY_SAVED_OBJECT_TYPE = void 0;
var _lodash = require("lodash");
var _elasticsearch = require("@elastic/elasticsearch");
var _constants = require("../../common/constants");
var _helpers = require("../../common/utils/helpers");
var _benchmark = require("../../common/schemas/benchmark");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PACKAGE_POLICY_SAVED_OBJECT_TYPE = 'ingest-package-policies';
exports.PACKAGE_POLICY_SAVED_OBJECT_TYPE = PACKAGE_POLICY_SAVED_OBJECT_TYPE;
const isFleetMissingAgentHttpError = error => error instanceof _elasticsearch.errors.ResponseError && error.statusCode === 404;
const isPolicyTemplate = input => _constants.SUPPORTED_POLICY_TEMPLATES.includes(input);
const getPackageNameQuery = () => {
  return `${PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:"${_constants.CLOUD_SECURITY_POSTURE_PACKAGE_NAME}"`;
};
const getPaginatedItems = (filteredItems, page, perPage) => {
  return filteredItems.slice((page - 1) * perPage, Math.min(filteredItems.length, page * perPage));
};
const getAgentStatusesByAgentPolicies = async (agentService, agentPolicies, logger) => {
  if (!(agentPolicies !== null && agentPolicies !== void 0 && agentPolicies.length)) return {};
  const internalAgentService = agentService.asInternalUser;
  const result = {};
  try {
    for (const agentPolicy of agentPolicies) {
      result[agentPolicy.id] = await internalAgentService.getAgentStatusForAgentPolicy(agentPolicy.id);
    }
  } catch (error) {
    if (isFleetMissingAgentHttpError(error)) {
      logger.debug('failed to get agent status for agent policy');
    } else {
      throw error;
    }
  }
  return result;
};
exports.getAgentStatusesByAgentPolicies = getAgentStatusesByAgentPolicies;
const getCspAgentPolicies = async (soClient, packagePolicies, agentPolicyService) => agentPolicyService.getByIds(soClient, (0, _lodash.uniq)((0, _lodash.map)(packagePolicies, 'policy_id')), {
  withPackagePolicies: true,
  ignoreMissing: true
});
exports.getCspAgentPolicies = getCspAgentPolicies;
const getCspPackagePolicies = async (soClient, packagePolicyService, packageName, queryParams, postureType, excludeVulnMgmtPackages = false) => {
  var _queryParams$sort_fie, _queryParams$page, _queryParams$per_page;
  const sortField = (_queryParams$sort_fie = queryParams.sort_field) === null || _queryParams$sort_fie === void 0 ? void 0 : _queryParams$sort_fie.replaceAll(_benchmark.BENCHMARK_PACKAGE_POLICY_PREFIX, '');
  const allCSPPackages = await packagePolicyService.list(soClient, {
    kuery: getPackageNameQuery(),
    page: 1,
    perPage: 10000,
    sortField,
    sortOrder: queryParams.sort_order
  });
  const filterPackagesByCriteria = input => {
    const showCSPMKSPMPackagesPolicies = input.enabled && (input.policy_template === _constants.KSPM_POLICY_TEMPLATE || input.policy_template === _constants.CSPM_POLICY_TEMPLATE);
    const showAllPackages = input.enabled;
    const showSelectedPostureTypePackages = input.enabled && input.policy_template === postureType;
    if (excludeVulnMgmtPackages) {
      return showCSPMKSPMPackagesPolicies;
    }
    if (postureType === 'all') {
      return showAllPackages;
    }
    return showSelectedPostureTypePackages;
  };
  const filteredItems = allCSPPackages.items.filter(pkg => pkg.inputs.filter(input => filterPackagesByCriteria(input)).length > 0 && (!queryParams.package_policy_name || pkg.name.toLowerCase().includes(queryParams.package_policy_name.toLowerCase())));
  const page = (_queryParams$page = queryParams === null || queryParams === void 0 ? void 0 : queryParams.page) !== null && _queryParams$page !== void 0 ? _queryParams$page : 1;
  const perPage = (_queryParams$per_page = queryParams === null || queryParams === void 0 ? void 0 : queryParams.per_page) !== null && _queryParams$per_page !== void 0 ? _queryParams$per_page : _benchmark.DEFAULT_BENCHMARKS_PER_PAGE;
  return {
    items: getPaginatedItems(filteredItems, page, perPage),
    total: filteredItems.length,
    page,
    perPage
  };
};
exports.getCspPackagePolicies = getCspPackagePolicies;
const getInstalledPolicyTemplates = async (packagePolicyClient, soClient) => {
  try {
    // getting all installed csp package policies
    const queryResult = await packagePolicyClient.list(soClient, {
      kuery: _helpers.CSP_FLEET_PACKAGE_KUERY,
      perPage: 1000
    });

    // getting installed policy templates by findings enabled inputs
    const enabledPolicyTemplates = queryResult.items.map(policy => {
      var _policy$inputs$find;
      return (_policy$inputs$find = policy.inputs.find(input => input.enabled)) === null || _policy$inputs$find === void 0 ? void 0 : _policy$inputs$find.policy_template;
    }).filter(isPolicyTemplate);
    // removing duplicates
    return [...new Set(enabledPolicyTemplates)];
  } catch (e) {
    return [];
  }
};
exports.getInstalledPolicyTemplates = getInstalledPolicyTemplates;