"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.regexStringRt = exports.paginationSchema = exports.limitedStringSchema = exports.limitedNumberSchema = exports.limitedArraySchema = exports.NonEmptyString = void 0;
var rt = _interopRequireWildcard(require("io-ts"));
var _Either = require("fp-ts/lib/Either");
var _constants = require("../constants");
var _types = require("./types");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const NonEmptyString = new rt.Type('NonEmptyString', rt.string.is, (input, context) => _Either.either.chain(rt.string.validate(input, context), s => {
  if (s.trim() !== '') {
    return rt.success(s);
  } else {
    return rt.failure(input, context, 'string must have length >= 1');
  }
}), rt.identity);
exports.NonEmptyString = NonEmptyString;
const limitedStringSchema = ({
  fieldName,
  min,
  max
}) => new rt.Type('LimitedString', rt.string.is, (input, context) => _Either.either.chain(rt.string.validate(input, context), s => {
  const trimmedString = s.trim();
  if (trimmedString.length === 0 && trimmedString.length < min) {
    return rt.failure(input, context, `The ${fieldName} field cannot be an empty string.`);
  }
  if (trimmedString.length < min) {
    return rt.failure(input, context, `The length of the ${fieldName} is too short. The minimum length is ${min}.`);
  }
  if (trimmedString.length > max) {
    return rt.failure(input, context, `The length of the ${fieldName} is too long. The maximum length is ${max}.`);
  }
  return rt.success(s);
}), rt.identity);
exports.limitedStringSchema = limitedStringSchema;
const limitedArraySchema = ({
  codec,
  fieldName,
  min,
  max
}) => new rt.Type('LimitedArray', input => rt.array(codec).is(input), (input, context) => _Either.either.chain(rt.array(codec).validate(input, context), s => {
  if (s.length < min) {
    return rt.failure(input, context, `The length of the field ${fieldName} is too short. Array must be of length >= ${min}.`);
  }
  if (s.length > max) {
    return rt.failure(input, context, `The length of the field ${fieldName} is too long. Array must be of length <= ${max}.`);
  }
  return rt.success(s);
}), rt.identity);
exports.limitedArraySchema = limitedArraySchema;
const paginationSchema = ({
  maxPerPage
}) => new rt.PartialType('Pagination', _types.PaginationSchemaRt.is, (u, c) => _Either.either.chain(_types.PaginationSchemaRt.validate(u, c), params => {
  var _params$page, _params$perPage;
  if (params.page == null && params.perPage == null) {
    return rt.success(params);
  }
  const pageAsNumber = (_params$page = params.page) !== null && _params$page !== void 0 ? _params$page : 0;
  const perPageAsNumber = (_params$perPage = params.perPage) !== null && _params$perPage !== void 0 ? _params$perPage : 0;
  if (perPageAsNumber > maxPerPage) {
    return rt.failure(u, c, `The provided perPage value is too high. The maximum allowed perPage value is ${maxPerPage}.`);
  }
  if (Math.max(pageAsNumber, pageAsNumber * perPageAsNumber) > _constants.MAX_DOCS_PER_PAGE) {
    return rt.failure(u, c, `The number of documents is too high. Paginating through more than ${_constants.MAX_DOCS_PER_PAGE} documents is not possible.`);
  }
  return rt.success({
    ...(params.page != null && {
      page: pageAsNumber
    }),
    ...(params.perPage != null && {
      perPage: perPageAsNumber
    })
  });
}), rt.identity, undefined);
exports.paginationSchema = paginationSchema;
const limitedNumberSchema = ({
  fieldName,
  min,
  max
}) => new rt.Type('LimitedNumber', rt.number.is, (input, context) => _Either.either.chain(rt.number.validate(input, context), s => {
  if (s < min) {
    return rt.failure(input, context, `The ${fieldName} field cannot be less than ${min}.`);
  }
  if (s > max) {
    return rt.failure(input, context, `The ${fieldName} field cannot be more than ${max}.`);
  }
  return rt.success(s);
}), rt.identity);
exports.limitedNumberSchema = limitedNumberSchema;
const regexStringRt = ({
  codec,
  pattern,
  message
}) => new rt.Type('RegexString', codec.is, (input, context) => _Either.either.chain(codec.validate(input, context), value => {
  const regex = new RegExp(pattern, 'g');
  if (!regex.test(value)) {
    return rt.failure(input, context, message);
  }
  return rt.success(value);
}), rt.identity);
exports.regexStringRt = regexStringRt;