"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.workpadServiceFactory = void 0;
var _constants = require("../../../common/lib/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
  Remove any top level keys from the workpad which will be rejected by validation
*/
const validKeys = ['@created', '@timestamp', 'assets', 'colors', 'css', 'variables', 'height', 'id', 'isWriteable', 'name', 'page', 'pages', 'width'];
const sanitizeWorkpad = function (workpad) {
  const workpadKeys = Object.keys(workpad);
  for (const key of workpadKeys) {
    if (!validKeys.includes(key)) {
      delete workpad[key];
    }
  }
  return workpad;
};
const workpadServiceFactory = ({
  coreStart,
  startPlugins
}) => {
  const getApiPath = function () {
    return `${_constants.API_ROUTE_WORKPAD}`;
  };
  return {
    get: async id => {
      const workpad = await coreStart.http.get(`${getApiPath()}/${id}`, {
        version: '1'
      });
      return {
        css: _constants.DEFAULT_WORKPAD_CSS,
        variables: [],
        ...workpad
      };
    },
    export: async id => {
      var _attributes$css, _attributes$variables;
      const workpad = await coreStart.http.get(`${getApiPath()}/export/${id}`, {
        version: '1'
      });
      const {
        attributes
      } = workpad;
      return {
        ...workpad,
        attributes: {
          ...attributes,
          css: (_attributes$css = attributes.css) !== null && _attributes$css !== void 0 ? _attributes$css : _constants.DEFAULT_WORKPAD_CSS,
          variables: (_attributes$variables = attributes.variables) !== null && _attributes$variables !== void 0 ? _attributes$variables : []
        }
      };
    },
    resolve: async id => {
      const {
        workpad,
        ...resolveProps
      } = await coreStart.http.get(`${getApiPath()}/resolve/${id}`, {
        version: '1'
      });
      return {
        ...resolveProps,
        workpad: {
          // @ts-ignore: Shimming legacy workpads that might not have CSS
          css: _constants.DEFAULT_WORKPAD_CSS,
          // @ts-ignore: Shimming legacy workpads that might not have variables
          variables: [],
          ...workpad
        }
      };
    },
    create: workpad => {
      return coreStart.http.post(getApiPath(), {
        body: JSON.stringify({
          ...sanitizeWorkpad({
            ...workpad
          }),
          assets: workpad.assets || {},
          variables: workpad.variables || []
        }),
        version: '1'
      });
    },
    import: workpad => coreStart.http.post(`${getApiPath()}/import`, {
      body: JSON.stringify({
        ...sanitizeWorkpad({
          ...workpad
        }),
        assets: workpad.assets || {},
        variables: workpad.variables || []
      }),
      version: '1'
    }),
    createFromTemplate: templateId => {
      return coreStart.http.post(getApiPath(), {
        body: JSON.stringify({
          templateId
        }),
        version: '1'
      });
    },
    findTemplates: async () => coreStart.http.get(_constants.API_ROUTE_TEMPLATES, {
      version: '1'
    }),
    find: searchTerm => {
      // TODO: this shouldn't be necessary.  Check for usage.
      const validSearchTerm = typeof searchTerm === 'string' && searchTerm.length > 0;
      return coreStart.http.get(`${getApiPath()}/find`, {
        query: {
          perPage: 10000,
          name: validSearchTerm ? searchTerm : ''
        },
        version: '1'
      });
    },
    remove: id => {
      return coreStart.http.delete(`${getApiPath()}/${id}`, {
        version: '1'
      });
    },
    update: (id, workpad) => {
      return coreStart.http.put(`${getApiPath()}/${id}`, {
        body: JSON.stringify({
          ...sanitizeWorkpad({
            ...workpad
          })
        }),
        version: '1'
      });
    },
    updateWorkpad: (id, workpad) => {
      return coreStart.http.put(`${_constants.API_ROUTE_WORKPAD_STRUCTURES}/${id}`, {
        body: JSON.stringify({
          ...sanitizeWorkpad({
            ...workpad
          })
        }),
        version: '1'
      });
    },
    updateAssets: (id, assets) => {
      return coreStart.http.put(`${_constants.API_ROUTE_WORKPAD_ASSETS}/${id}`, {
        body: JSON.stringify(assets),
        version: '1'
      });
    },
    getRuntimeZip: workpad => {
      return coreStart.http.post(_constants.API_ROUTE_SHAREABLE_ZIP, {
        body: JSON.stringify(workpad),
        version: '1'
      });
    }
  };
};
exports.workpadServiceFactory = workpadServiceFactory;