"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isSummaryActionThrottled = exports.isSummaryActionOnInterval = exports.isSummaryAction = exports.isActionOnInterval = exports.getSummaryActionsFromTaskState = exports.getSummaryActionTimeBounds = exports.generateActionHash = void 0;
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isSummaryAction = action => {
  var _action$frequency;
  return (action === null || action === void 0 ? void 0 : (_action$frequency = action.frequency) === null || _action$frequency === void 0 ? void 0 : _action$frequency.summary) || false;
};
exports.isSummaryAction = isSummaryAction;
const isActionOnInterval = action => {
  if (!(action !== null && action !== void 0 && action.frequency)) {
    return false;
  }
  return action.frequency.notifyWhen === _common.RuleNotifyWhenTypeValues[2] && typeof action.frequency.throttle === 'string';
};
exports.isActionOnInterval = isActionOnInterval;
const isSummaryActionOnInterval = action => {
  var _action$frequency2;
  return isActionOnInterval(action) && ((_action$frequency2 = action.frequency) === null || _action$frequency2 === void 0 ? void 0 : _action$frequency2.summary);
};
exports.isSummaryActionOnInterval = isSummaryActionOnInterval;
const isSummaryActionThrottled = ({
  action,
  throttledSummaryActions,
  logger
}) => {
  if (!isActionOnInterval(action)) {
    return false;
  }
  if (!throttledSummaryActions) {
    return false;
  }
  const throttledAction = throttledSummaryActions[action === null || action === void 0 ? void 0 : action.uuid];
  if (!throttledAction) {
    return false;
  }
  let throttleMills = 0;
  try {
    throttleMills = (0, _common.parseDuration)(action === null || action === void 0 ? void 0 : action.frequency.throttle);
  } catch (e) {
    logger.debug(`Action'${action === null || action === void 0 ? void 0 : action.actionTypeId}:${action === null || action === void 0 ? void 0 : action.id}', has an invalid throttle interval`);
  }
  const throttled = new Date(throttledAction.date).getTime() + throttleMills > Date.now();
  if (throttled) {
    logger.debug(`skipping scheduling the action '${action === null || action === void 0 ? void 0 : action.actionTypeId}:${action === null || action === void 0 ? void 0 : action.id}', summary action is still being throttled`);
  }
  return throttled;
};
exports.isSummaryActionThrottled = isSummaryActionThrottled;
const generateActionHash = action => {
  var _action$frequency3, _action$frequency4;
  return `${(action === null || action === void 0 ? void 0 : action.actionTypeId) || 'no-action-type-id'}:${action !== null && action !== void 0 && (_action$frequency3 = action.frequency) !== null && _action$frequency3 !== void 0 && _action$frequency3.summary ? 'summary' : (action === null || action === void 0 ? void 0 : action.group) || 'no-action-group'}:${(action === null || action === void 0 ? void 0 : (_action$frequency4 = action.frequency) === null || _action$frequency4 === void 0 ? void 0 : _action$frequency4.throttle) || 'no-throttling'}`;
};
exports.generateActionHash = generateActionHash;
const getSummaryActionsFromTaskState = ({
  actions,
  summaryActions = {}
}) => {
  return Object.entries(summaryActions).reduce((newObj, [key, val]) => {
    const actionExists = actions.find(action => {
      var _action$frequency5;
      return ((_action$frequency5 = action.frequency) === null || _action$frequency5 === void 0 ? void 0 : _action$frequency5.summary) && (action.uuid === key || generateActionHash(action) === key);
    });
    if (actionExists) {
      // replace hash with uuid
      newObj[actionExists.uuid] = val;
    }
    return newObj;
  }, {});
};
exports.getSummaryActionsFromTaskState = getSummaryActionsFromTaskState;
const getSummaryActionTimeBounds = (action, ruleSchedule, previousStartedAt) => {
  if (!isSummaryAction(action)) {
    return {
      start: undefined,
      end: undefined
    };
  }
  let startDate;
  const now = Date.now();
  if (isActionOnInterval(action)) {
    // If action is throttled, set time bounds using throttle interval
    const throttleMills = (0, _common.parseDuration)(action.frequency.throttle);
    startDate = new Date(now - throttleMills);
  } else {
    // If action is not throttled, set time bounds to previousStartedAt - now
    // If previousStartedAt is null, use the rule schedule interval
    if (previousStartedAt) {
      startDate = previousStartedAt;
    } else {
      const scheduleMillis = (0, _common.parseDuration)(ruleSchedule.interval);
      startDate = new Date(now - scheduleMillis);
    }
  }
  return {
    start: startDate.valueOf(),
    end: now.valueOf()
  };
};
exports.getSummaryActionTimeBounds = getSummaryActionTimeBounds;