"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createOrUpdateComponentTemplate = void 0;
var _std = require("@kbn/std");
var _retry_transient_es_errors = require("./retry_transient_es_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getIndexTemplatesUsingComponentTemplate = async (esClient, componentTemplateName, totalFieldsLimit, logger) => {
  // Get all index templates and filter down to just the ones referencing this component template
  const {
    index_templates: indexTemplates
  } = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.getIndexTemplate(), {
    logger
  });
  const indexTemplatesUsingComponentTemplate = (indexTemplates !== null && indexTemplates !== void 0 ? indexTemplates : []).filter(indexTemplate => {
    if (indexTemplate && indexTemplate.index_template && indexTemplate.index_template.composed_of) {
      return indexTemplate.index_template.composed_of.includes(componentTemplateName);
    }
    return false;
  });
  await (0, _std.asyncForEach)(indexTemplatesUsingComponentTemplate, async template => {
    await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => {
      var _template$index_templ;
      return esClient.indices.putIndexTemplate({
        name: template.name,
        body: {
          ...template.index_template,
          template: {
            ...template.index_template.template,
            settings: {
              ...((_template$index_templ = template.index_template.template) === null || _template$index_templ === void 0 ? void 0 : _template$index_templ.settings),
              'index.mapping.total_fields.limit': totalFieldsLimit
            }
          }
        }
      });
    }, {
      logger
    });
  });
};
const createOrUpdateComponentTemplateHelper = async (esClient, template, totalFieldsLimit, logger) => {
  try {
    await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.cluster.putComponentTemplate(template), {
      logger
    });
  } catch (error) {
    var _error$meta, _error$meta$body, _error$meta$body$erro, _error$meta$body$erro2, _error$meta$body$erro3, _error$meta$body$erro4;
    const reason = error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : (_error$meta$body$erro2 = _error$meta$body$erro.caused_by) === null || _error$meta$body$erro2 === void 0 ? void 0 : (_error$meta$body$erro3 = _error$meta$body$erro2.caused_by) === null || _error$meta$body$erro3 === void 0 ? void 0 : (_error$meta$body$erro4 = _error$meta$body$erro3.caused_by) === null || _error$meta$body$erro4 === void 0 ? void 0 : _error$meta$body$erro4.reason;
    if (reason && reason.match(/Limit of total fields \[\d+\] has been exceeded/) != null) {
      // This error message occurs when there is an index template using this component template
      // that contains a field limit setting that using this component template exceeds
      // Specifically, this can happen for the ECS component template when we add new fields
      // to adhere to the ECS spec. Individual index templates specify field limits so if the
      // number of new ECS fields pushes the composed mapping above the limit, this error will
      // occur. We have to update the field limit inside the index template now otherwise we
      // can never update the component template
      await getIndexTemplatesUsingComponentTemplate(esClient, template.name, totalFieldsLimit, logger);

      // Try to update the component template again
      await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.cluster.putComponentTemplate(template), {
        logger
      });
    } else {
      throw error;
    }
  }
};
const createOrUpdateComponentTemplate = async ({
  logger,
  esClient,
  template,
  totalFieldsLimit
}) => {
  logger.info(`Installing component template ${template.name}`);
  try {
    await createOrUpdateComponentTemplateHelper(esClient, template, totalFieldsLimit, logger);
  } catch (err) {
    logger.error(`Error installing component template ${template.name} - ${err.message}`);
    throw err;
  }
};
exports.createOrUpdateComponentTemplate = createOrUpdateComponentTemplate;