"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTopTermRequest = exports.fetchTopTerms = void 0;
var _lodash = require("lodash");
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _mlRandomSamplerUtils = require("@kbn/ml-random-sampler-utils");
var _constants = require("../../../../common/constants");
var _is_request_aborted_error = require("../../../lib/is_request_aborted_error");
var _get_query_with_params = require("./get_query_with_params");
var _get_request_base = require("./get_request_base");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// TODO Consolidate with duplicate `fetchDurationFieldCandidates` in
// `x-pack/plugins/apm/server/routes/correlations/queries/fetch_failed_events_correlation_p_values.ts`

const getTopTermRequest = (params, fieldName, {
  wrap
}) => {
  var _params$timeFieldName;
  const query = (0, _get_query_with_params.getQueryWithParams)({
    params
  });
  const timeFieldName = (_params$timeFieldName = params.timeFieldName) !== null && _params$timeFieldName !== void 0 ? _params$timeFieldName : '@timestamp';
  let filter = [];
  if (query.bool && Array.isArray(query.bool.filter)) {
    filter = query.bool.filter.filter(d => Object.keys(d)[0] !== 'range');
    query.bool.filter = [...filter, {
      range: {
        [timeFieldName]: {
          gte: params.deviationMin,
          lt: params.deviationMax,
          format: 'epoch_millis'
        }
      }
    }];
  }
  const termAgg = {
    log_rate_top_terms: {
      terms: {
        field: fieldName,
        size: 10
      }
    }
  };
  const body = {
    query,
    size: 0,
    aggs: wrap(termAgg)
  };
  return {
    ...(0, _get_request_base.getRequestBase)(params),
    body
  };
};
exports.getTopTermRequest = getTopTermRequest;
const fetchTopTerms = async (esClient, params, fieldNames, logger,
// The default value of 1 means no sampling will be used
sampleProbability = 1, emitError, abortSignal) => {
  const randomSamplerWrapper = (0, _mlRandomSamplerUtils.createRandomSamplerWrapper)({
    probability: sampleProbability,
    seed: _constants.RANDOM_SAMPLER_SEED
  });
  const result = [];
  const settledPromises = await Promise.allSettled(fieldNames.map(fieldName => esClient.search(getTopTermRequest(params, fieldName, randomSamplerWrapper), {
    signal: abortSignal,
    maxRetries: 0
  })));
  function reportError(fieldName, error) {
    if (!(0, _is_request_aborted_error.isRequestAbortedError)(error)) {
      logger.error(`Failed to fetch term aggregation for fieldName "${fieldName}", got: \n${JSON.stringify(error, null, 2)}`);
      emitError(`Failed to fetch term aggregation for fieldName "${fieldName}".`);
    }
  }
  for (const [index, settledPromise] of settledPromises.entries()) {
    const fieldName = fieldNames[index];
    if (settledPromise.status === 'rejected') {
      reportError(fieldName, settledPromise.reason);
      // Still continue the analysis even if individual p-value queries fail.
      continue;
    }
    const resp = settledPromise.value;
    if (resp.aggregations === undefined) {
      reportError(fieldName, resp);
      // Still continue the analysis even if individual p-value queries fail.
      continue;
    }
    const overallResult = randomSamplerWrapper.unwrap(resp.aggregations).log_rate_top_terms;
    for (const bucket of overallResult.buckets) {
      result.push({
        key: `${fieldName}:${String(bucket.key)}`,
        type: _mlAggUtils.SIGNIFICANT_ITEM_TYPE.KEYWORD,
        fieldName,
        fieldValue: String(bucket.key),
        doc_count: bucket.doc_count,
        bg_count: 0,
        total_doc_count: 0,
        total_bg_count: 0,
        score: 0,
        pValue: 1,
        normalizedScore: 0
      });
    }
  }
  return (0, _lodash.uniqBy)(result, d => `${d.fieldName},${d.fieldValue}`);
};
exports.fetchTopTerms = fetchTopTerms;