"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.shouldTrackGenAiToken = exports.getGenAiTokenTracking = void 0;
var _stream = require("stream");
var _get_token_count_from_bedrock_invoke = require("./get_token_count_from_bedrock_invoke");
var _get_token_count_from_openai_stream = require("./get_token_count_from_openai_stream");
var _get_token_count_from_invoke_stream = require("./get_token_count_from_invoke_stream");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * Calculates the total, prompt, and completion token counts from different types of responses.
 * It handles both streamed and non-streamed responses from OpenAI and Bedrock.
 * It returns null if it cannot calculate the token counts.
 * @param actionTypeId the action type id
 * @param logger the logger
 * @param result the result from the action executor
 * @param validatedParams the validated params from the action executor
 */
const getGenAiTokenTracking = async ({
  actionTypeId,
  logger,
  result,
  validatedParams
}) => {
  // this is a streamed OpenAI or Bedrock response, using the subAction invokeStream to stream the response as a simple string
  if (validatedParams.subAction === 'invokeStream' && result.data instanceof _stream.Readable) {
    try {
      const {
        total,
        prompt,
        completion
      } = await (0, _get_token_count_from_invoke_stream.getTokenCountFromInvokeStream)({
        responseStream: result.data.pipe(new _stream.PassThrough()),
        actionTypeId,
        body: validatedParams.subActionParams,
        logger
      });
      return {
        total_tokens: total,
        prompt_tokens: prompt,
        completion_tokens: completion
      };
    } catch (e) {
      logger.error('Failed to calculate tokens from Invoke Stream subaction streaming response');
      logger.error(e);
    }
  }

  // this is a streamed OpenAI response, which did not use the subAction invokeStream
  if (actionTypeId === '.gen-ai' && result.data instanceof _stream.Readable) {
    try {
      const {
        total,
        prompt,
        completion
      } = await (0, _get_token_count_from_openai_stream.getTokenCountFromOpenAIStream)({
        responseStream: result.data.pipe(new _stream.PassThrough()),
        body: validatedParams.subActionParams.body,
        logger
      });
      return {
        total_tokens: total,
        prompt_tokens: prompt,
        completion_tokens: completion
      };
    } catch (e) {
      logger.error('Failed to calculate tokens from streaming response');
      logger.error(e);
    }
  }

  // this is a non-streamed OpenAI response, which comes with the usage object
  if (actionTypeId === '.gen-ai') {
    var _data$usage$total_tok, _data$usage, _data$usage$prompt_to, _data$usage2, _data$usage$completio, _data$usage3;
    const data = result.data;
    if (data.usage == null) {
      logger.error('Response did not contain usage object');
      return null;
    }
    return {
      total_tokens: (_data$usage$total_tok = (_data$usage = data.usage) === null || _data$usage === void 0 ? void 0 : _data$usage.total_tokens) !== null && _data$usage$total_tok !== void 0 ? _data$usage$total_tok : 0,
      prompt_tokens: (_data$usage$prompt_to = (_data$usage2 = data.usage) === null || _data$usage2 === void 0 ? void 0 : _data$usage2.prompt_tokens) !== null && _data$usage$prompt_to !== void 0 ? _data$usage$prompt_to : 0,
      completion_tokens: (_data$usage$completio = (_data$usage3 = data.usage) === null || _data$usage3 === void 0 ? void 0 : _data$usage3.completion_tokens) !== null && _data$usage$completio !== void 0 ? _data$usage$completio : 0
    };
  }

  // this is a non-streamed Bedrock response
  if (actionTypeId === '.bedrock' && (validatedParams.subAction === 'run' || validatedParams.subAction === 'test')) {
    try {
      const {
        total,
        prompt,
        completion
      } = await (0, _get_token_count_from_bedrock_invoke.getTokenCountFromBedrockInvoke)({
        response: result.data.completion,
        body: validatedParams.subActionParams.body
      });
      return {
        total_tokens: total,
        prompt_tokens: prompt,
        completion_tokens: completion
      };
    } catch (e) {
      logger.error('Failed to calculate tokens from Bedrock invoke response');
      logger.error(e);
    }
  }

  // this is a non-streamed Bedrock response used by security solution
  if (actionTypeId === '.bedrock' && validatedParams.subAction === 'invokeAI') {
    try {
      const {
        total,
        prompt,
        completion
      } = await (0, _get_token_count_from_bedrock_invoke.getTokenCountFromBedrockInvoke)({
        response: result.data.message,
        body: JSON.stringify({
          prompt: validatedParams.subActionParams.messages[0].content
        })
      });
      return {
        total_tokens: total,
        prompt_tokens: prompt,
        completion_tokens: completion
      };
    } catch (e) {
      logger.error('Failed to calculate tokens from Bedrock invoke response');
      logger.error(e);
    }
  }
  return null;
};
exports.getGenAiTokenTracking = getGenAiTokenTracking;
const shouldTrackGenAiToken = actionTypeId => actionTypeId === '.gen-ai' || actionTypeId === '.bedrock';
exports.shouldTrackGenAiToken = shouldTrackGenAiToken;