"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TMSService = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _ems_service = require("./ems_service");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
                                                                                                                                                                                                                                                                                                                                                                                           * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
                                                                                                                                                                                                                                                                                                                                                                                           * or more contributor license agreements. Licensed under the Elastic License
                                                                                                                                                                                                                                                                                                                                                                                           * 2.0; you may not use this file except in compliance with the Elastic License
                                                                                                                                                                                                                                                                                                                                                                                           * 2.0.
                                                                                                                                                                                                                                                                                                                                                                                           */
class TMSService extends _ems_service.AbstractEmsService {
  constructor(config, emsClient, proxyPath) {
    super(config, emsClient, proxyPath);
    _defineProperty(this, "_config", void 0);
    _defineProperty(this, "_getRasterStyleJson", _lodash.default.once(async () => {
      const rasterUrl = this._getStyleUrlForLocale('raster');
      if (rasterUrl) {
        const url = this._proxyPath + this._getAbsoluteUrl(rasterUrl);
        return this._emsClient.getManifest(this._emsClient.extendUrlWithParams(url));
      } else {
        return;
      }
    }));
    _defineProperty(this, "_getVectorStyleJsonRaw", _lodash.default.once(async () => {
      const vectorUrl = this._getStyleUrlForLocale('vector');
      if (vectorUrl) {
        const url = this._proxyPath + this._getAbsoluteUrl(vectorUrl);
        const vectorJson = await this._emsClient.getManifest(this._emsClient.extendUrlWithParams(url));
        return {
          ...vectorJson
        };
      } else {
        return;
      }
    }));
    _defineProperty(this, "_getVectorStyleJsonInlined", _lodash.default.once(async () => {
      const vectorJson = await this._getVectorStyleJsonRaw();
      if (vectorJson) {
        const inlinedSources = {};
        const {
          sources
        } = vectorJson;
        for (const sourceName of Object.getOwnPropertyNames(sources)) {
          const {
            url
          } = sources[sourceName];
          const sourceUrl = this._proxyPath + this._getAbsoluteUrl(url);
          const extendedUrl = this._emsClient.extendUrlWithParams(sourceUrl);
          const sourceJson = await this._emsClient.getManifest(extendedUrl);
          const extendedTileUrls = sourceJson.tiles.map(tileUrl => {
            const url = this._proxyPath + this._getAbsoluteUrl(tileUrl);
            return this._emsClient.extendUrlWithParams(url);
          });
          inlinedSources[sourceName] = {
            ...sourceJson,
            type: 'vector',
            tiles: extendedTileUrls
          };
        }
        return {
          ...vectorJson,
          sources: inlinedSources,
          sprite: await this._getSpriteSheetRootPath(),
          glyphs: await this._getUrlTemplateForGlyphs()
        };
      } else {
        return;
      }
    }));
    this._config = config;
  }
  async getDefaultRasterStyle() {
    const tileJson = await this._getRasterStyleJson();
    if (tileJson) {
      const tiles = tileJson.tiles.map(tile => this._proxyPath + this._getAbsoluteUrl(tile));
      return {
        ...tileJson,
        ...{
          tiles
        }
      };
    } else {
      return;
    }
  }
  async getUrlTemplate() {
    const tileJson = await this._getRasterStyleJson();
    if (tileJson) {
      const directUrl = this._proxyPath + this._getAbsoluteUrl(tileJson.tiles[0]);
      return this._emsClient.extendUrlWithParams(directUrl);
    } else {
      return '';
    }
  }
  async getUrlTemplateForVector(sourceId) {
    const tileJson = await this._getVectorStyleJsonInlined();
    if (!tileJson) {
      return '';
    }
    if (tileJson.sources[sourceId] && tileJson.sources[sourceId].tiles) {
      const directUrl = this._proxyPath + this._getAbsoluteUrl(tileJson.sources[sourceId].tiles[0]);
      return this._emsClient.extendUrlWithParams(directUrl);
    } else {
      return '';
    }
  }
  async getVectorStyleSheet() {
    return await this._getVectorStyleJsonInlined();
  }
  async getVectorStyleSheetRaw() {
    return await this._getVectorStyleJsonRaw();
  }
  async getSpriteSheetMeta(isRetina = false) {
    const metaUrl = await this.getSpriteSheetJsonPath(isRetina);
    const spritePngs = await this.getSpriteSheetPngPath(isRetina);
    if (metaUrl && spritePngs) {
      const metaUrlExtended = this._emsClient.extendUrlWithParams(metaUrl);
      const jsonMeta = await this._emsClient.getManifest(metaUrlExtended);
      return {
        png: spritePngs,
        json: jsonMeta
      };
    } else {
      return;
    }
  }
  async getSpriteSheetJsonPath(isRetina = false) {
    const spriteSheetRootPath = await this._getSpriteSheetRootPath();
    if (spriteSheetRootPath) {
      const suffix = isRetina ? '@2x' : '';
      return spriteSheetRootPath + suffix + '.json';
    } else {
      return '';
    }
  }
  async getSpriteSheetPngPath(isRetina = false) {
    const spriteSheetRootPath = await this._getSpriteSheetRootPath();
    if (spriteSheetRootPath) {
      const suffix = isRetina ? '@2x' : '';
      return spriteSheetRootPath + suffix + '.png';
    } else {
      return '';
    }
  }
  getDisplayName() {
    return this._emsClient.getValueInLanguage(this._config.name);
  }
  async getMinZoom() {
    const tileJson = await this._getRasterStyleJson();
    if (tileJson) {
      return tileJson.minzoom;
    } else {
      return;
    }
  }
  async getMaxZoom() {
    const tileJson = await this._getRasterStyleJson();
    if (tileJson) {
      return tileJson.maxzoom;
    } else {
      return;
    }
  }
  getId() {
    return this._config.id;
  }
  hasId(id) {
    return this._config.id === id;
  }
  getApiUrl() {
    return this._emsClient.getTileApiUrl();
  }
  _getStyleUrlForLocale(formatType) {
    let vectorFormats = this._getFormats(formatType, this._emsClient.getLocale());
    if (!vectorFormats.length) {
      //fallback to default locale
      vectorFormats = this._getFormats(formatType, this._emsClient.getDefaultLocale());
    }
    if (!vectorFormats.length) {
      // eslint-disable-next-line max-len
      throw new Error(`Cannot find ${formatType} tile layer for locale ${this._emsClient.getLocale()} or ${this._emsClient.getDefaultLocale()}`);
    }
    const defaultStyle = vectorFormats[0];
    if (defaultStyle && defaultStyle.hasOwnProperty('url')) {
      return defaultStyle.url;
    }
  }
  _getFormats(formatType, locale) {
    return this._config.formats.filter(format => format.locale === locale && format.format === formatType);
  }
  async _getSpriteSheetRootPath() {
    const vectorStyleJson = await this._getVectorStyleJsonRaw();
    if (vectorStyleJson) {
      return this._proxyPath + this._getAbsoluteUrl(vectorStyleJson.sprite);
    } else {
      return '';
    }
  }
  async _getUrlTemplateForGlyphs() {
    const vectorStyleJson = await this._getVectorStyleJsonRaw();
    if (vectorStyleJson) {
      return this._proxyPath + this._getAbsoluteUrl(vectorStyleJson.glyphs);
    } else {
      return '';
    }
  }
}
exports.TMSService = TMSService;