import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
import _typeof from "@babel/runtime/helpers/typeof";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import { left, fold, right } from 'fp-ts/lib/Either';
import { pipe } from 'fp-ts/lib/pipeable';
import { isObject, get } from 'lodash/fp';

/**
 * Given an original object and a decoded object this will return an error
 * if and only if the original object has additional keys that the decoded
 * object does not have. If the original decoded already has an error, then
 * this will return the error as is and not continue.
 *
 * NOTE: You MUST use t.exact(...) for this to operate correctly as your schema
 * needs to remove additional keys before the compare
 *
 * You might not need this in the future if the below issue is solved:
 * https://github.com/gcanti/io-ts/issues/322
 *
 * @param original The original to check if it has additional keys
 * @param decoded The decoded either which has either an existing error or the
 * decoded object which could have additional keys stripped from it.
 */
export var exactCheck = function exactCheck(original, decoded) {
  var onLeft = function onLeft(errors) {
    return left(errors);
  };
  var onRight = function onRight(decodedValue) {
    var differences = findDifferencesRecursive(original, decodedValue);
    if (differences.length !== 0) {
      var validationError = {
        value: differences,
        context: [],
        message: "invalid keys \"".concat(differences.join(','), "\"")
      };
      var error = [validationError];
      return left(error);
    } else {
      return right(decodedValue);
    }
  };
  return pipe(decoded, fold(onLeft, onRight));
};
export var findDifferencesRecursive = function findDifferencesRecursive(original, decodedValue) {
  if (decodedValue === null && original === null) {
    // both the decodedValue and the original are null which indicates that they are equal
    // so do not report differences
    return [];
  } else if (decodedValue == null) {
    try {
      // It is null and painful when the original contains an object or an array
      // the the decoded value does not have.
      return [JSON.stringify(original)];
    } catch (err) {
      return ['circular reference'];
    }
  } else if (_typeof(original) !== 'object' || original == null) {
    // We are not an object or null so do not report differences
    return [];
  } else {
    var decodedKeys = Object.keys(decodedValue);
    var differences = Object.keys(original).flatMap(function (originalKey) {
      var foundKey = decodedKeys.some(function (key) {
        return key === originalKey;
      });
      var topLevelKey = foundKey ? [] : [originalKey];
      // I use lodash to cheat and get an any (not going to lie ;-))
      var valueObjectOrArrayOriginal = get(originalKey, original);
      var valueObjectOrArrayDecoded = get(originalKey, decodedValue);
      if (isObject(valueObjectOrArrayOriginal)) {
        return [].concat(topLevelKey, _toConsumableArray(findDifferencesRecursive(valueObjectOrArrayOriginal, valueObjectOrArrayDecoded)));
      } else if (Array.isArray(valueObjectOrArrayOriginal)) {
        return [].concat(topLevelKey, _toConsumableArray(valueObjectOrArrayOriginal.flatMap(function (arrayElement, index) {
          return findDifferencesRecursive(arrayElement, get(index, valueObjectOrArrayDecoded));
        })));
      } else {
        return topLevelKey;
      }
    });
    return differences;
  }
};