"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LogRotator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var chokidar = _interopRequireWildcard(require("chokidar"));
var _fs = _interopRequireDefault(require("fs"));
var _lodash = require("lodash");
var _os = require("os");
var _path = require("path");
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _util = require("util");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const mkdirAsync = (0, _util.promisify)(_fs.default.mkdir);
const readdirAsync = (0, _util.promisify)(_fs.default.readdir);
const renameAsync = (0, _util.promisify)(_fs.default.rename);
const statAsync = (0, _util.promisify)(_fs.default.stat);
const unlinkAsync = (0, _util.promisify)(_fs.default.unlink);
const writeFileAsync = (0, _util.promisify)(_fs.default.writeFile);
class LogRotator {
  constructor(config, server) {
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "logFilePath", void 0);
    (0, _defineProperty2.default)(this, "everyBytes", void 0);
    (0, _defineProperty2.default)(this, "keepFiles", void 0);
    (0, _defineProperty2.default)(this, "running", void 0);
    (0, _defineProperty2.default)(this, "logFileSize", void 0);
    (0, _defineProperty2.default)(this, "isRotating", void 0);
    (0, _defineProperty2.default)(this, "throttledRotate", void 0);
    (0, _defineProperty2.default)(this, "stalker", void 0);
    (0, _defineProperty2.default)(this, "usePolling", void 0);
    (0, _defineProperty2.default)(this, "pollingInterval", void 0);
    (0, _defineProperty2.default)(this, "stalkerUsePollingPolicyTestTimeout", void 0);
    (0, _defineProperty2.default)(this, "shouldUsePolling", void 0);
    (0, _defineProperty2.default)(this, "stop", () => {
      if (!this.running) {
        return;
      }

      // cleanup exit listener
      this._deleteExitListener();

      // stop log file size monitor
      this._stopLogFileSizeMonitor();
      this.running = false;
    });
    (0, _defineProperty2.default)(this, "_logFileSizeMonitorHandler", async (filename, stats) => {
      if (!filename || !stats) {
        return;
      }
      this.logFileSize = stats.size || 0;
      await this.throttledRotate();
    });
    this.config = config;
    this.log = server.log.bind(server);
    this.logFilePath = config.dest;
    this.everyBytes = config.rotate.everyBytes;
    this.keepFiles = config.rotate.keepFiles;
    this.running = false;
    this.logFileSize = 0;
    this.isRotating = false;
    this.throttledRotate = (0, _lodash.throttle)(async () => await this._rotate(), 5000);
    this.stalker = null;
    this.usePolling = config.rotate.usePolling;
    this.pollingInterval = config.rotate.pollingInterval;
    this.shouldUsePolling = false;
    this.stalkerUsePollingPolicyTestTimeout = null;
  }
  async start() {
    if (this.running) {
      return;
    }
    this.running = true;

    // create exit listener for cleanup purposes
    this._createExitListener();

    // call rotate on startup
    await this._callRotateOnStartup();

    // init log file size monitor
    await this._startLogFileSizeMonitor();
  }
  async _shouldUsePolling() {
    try {
      // Setup a test file in order to try the fs env
      // and understand if we need to usePolling or not
      const tempFileDir = (0, _os.tmpdir)();
      const tempFile = (0, _path.join)(tempFileDir, 'kbn_log_rotation_use_polling_test_file.log');
      await mkdirAsync(tempFileDir, {
        recursive: true
      });
      await writeFileAsync(tempFile, '');

      // setup fs.watch for the temp test file
      const testWatcher = _fs.default.watch(tempFile, {
        persistent: false
      });

      // await writeFileAsync(tempFile, 'test');

      const usePollingTest$ = new _rxjs.Observable(observer => {
        // observable complete function
        const completeFn = completeStatus => {
          if (this.stalkerUsePollingPolicyTestTimeout) {
            clearTimeout(this.stalkerUsePollingPolicyTestTimeout);
          }
          testWatcher.close();
          observer.next(completeStatus);
          observer.complete();
        };

        // setup conditions that would fire the observable
        this.stalkerUsePollingPolicyTestTimeout = setTimeout(() => completeFn(true), this.config.rotate.pollingPolicyTestTimeout || 15000);
        testWatcher.on('change', () => completeFn(false));
        testWatcher.on('error', () => completeFn(true));

        // fire test watcher events
        setTimeout(() => {
          _fs.default.writeFileSync(tempFile, 'test');
        }, 0);
      });

      // wait for the first observable result and consider it as the result
      // for our use polling test
      const usePollingTestResult = await usePollingTest$.pipe((0, _operators.first)()).toPromise();

      // delete the temp file used for the test
      await unlinkAsync(tempFile);
      return usePollingTestResult;
    } catch {
      return true;
    }
  }
  async _startLogFileSizeMonitor() {
    this.usePolling = this.config.rotate.usePolling;
    this.shouldUsePolling = await this._shouldUsePolling();
    if (this.usePolling && !this.shouldUsePolling) {
      this.log(['warning', 'logging:rotate'], 'Looks like your current environment support a faster algorithm than polling. You can try to disable `usePolling`');
    }
    if (!this.usePolling && this.shouldUsePolling) {
      this.log(['error', 'logging:rotate'], 'Looks like within your current environment you need to use polling in order to enable log rotator. Please enable `usePolling`');
    }
    this.stalker = chokidar.watch(this.logFilePath, {
      ignoreInitial: true,
      awaitWriteFinish: false,
      useFsEvents: false,
      usePolling: this.usePolling,
      interval: this.pollingInterval,
      binaryInterval: this.pollingInterval,
      alwaysStat: true,
      atomic: false
    });
    this.stalker.on('change', this._logFileSizeMonitorHandler);
  }
  _stopLogFileSizeMonitor() {
    if (!this.stalker) {
      return;
    }
    this.stalker.close();
    if (this.stalkerUsePollingPolicyTestTimeout) {
      clearTimeout(this.stalkerUsePollingPolicyTestTimeout);
    }
  }
  _createExitListener() {
    process.on('exit', this.stop);
  }
  _deleteExitListener() {
    process.removeListener('exit', this.stop);
  }
  async _getLogFileSizeAndCreateIfNeeded() {
    try {
      const logFileStats = await statAsync(this.logFilePath);
      return logFileStats.size;
    } catch {
      // touch the file to make the watcher being able to register
      // change events
      await writeFileAsync(this.logFilePath, '');
      return 0;
    }
  }
  async _callRotateOnStartup() {
    this.logFileSize = await this._getLogFileSizeAndCreateIfNeeded();
    await this._rotate();
  }
  _shouldRotate() {
    // should rotate evaluation
    // 1. should rotate if current log size exceeds
    //    the defined one on everyBytes
    // 2. should not rotate if is already rotating or if any
    //    of the conditions on 1. do not apply
    if (this.isRotating) {
      return false;
    }
    return this.logFileSize >= this.everyBytes;
  }
  async _rotate() {
    if (!this._shouldRotate()) {
      return;
    }
    await this._rotateNow();
  }
  async _rotateNow() {
    // rotate process
    // 1. get rotated files metadata (list of log rotated files present on the log folder, numerical sorted)
    // 2. delete last file
    // 3. rename all files to the correct index +1
    // 4. rename + compress current log into 1
    // 5. send SIGHUP to reload log config

    // rotate process is starting
    this.isRotating = true;

    // get rotated files metadata
    const foundRotatedFiles = await this._readRotatedFilesMetadata();

    // delete number of rotated files exceeding the keepFiles limit setting
    const rotatedFiles = await this._deleteFoundRotatedFilesAboveKeepFilesLimit(foundRotatedFiles);

    // delete last file
    await this._deleteLastRotatedFile(rotatedFiles);

    // rename all files to correct index + 1
    // and normalize numbering if by some reason
    // (for example log file deletion) that numbering
    // was interrupted
    await this._renameRotatedFilesByOne(rotatedFiles);

    // rename current log into 0
    await this._rotateCurrentLogFile();

    // send SIGHUP to reload log configuration
    this._sendReloadLogConfigSignal();

    // Reset log file size
    this.logFileSize = 0;

    // rotate process is finished
    this.isRotating = false;
  }
  async _readRotatedFilesMetadata() {
    const logFileBaseName = (0, _path.basename)(this.logFilePath);
    const logFilesFolder = (0, _path.dirname)(this.logFilePath);
    const foundLogFiles = await readdirAsync(logFilesFolder);
    return foundLogFiles.filter(file => new RegExp(`${logFileBaseName}\\.\\d`).test(file))
    // we use .slice(-1) here in order to retrieve the last number match in the read filenames
    .sort((a, b) => Number(a.match(/(\d+)/g).slice(-1)) - Number(b.match(/(\d+)/g).slice(-1))).map(filename => `${logFilesFolder}${_path.sep}${filename}`);
  }
  async _deleteFoundRotatedFilesAboveKeepFilesLimit(foundRotatedFiles) {
    if (foundRotatedFiles.length <= this.keepFiles) {
      return foundRotatedFiles;
    }
    const finalRotatedFiles = foundRotatedFiles.slice(0, this.keepFiles);
    const rotatedFilesToDelete = foundRotatedFiles.slice(finalRotatedFiles.length, foundRotatedFiles.length);
    await Promise.all(rotatedFilesToDelete.map(rotatedFilePath => unlinkAsync(rotatedFilePath)));
    return finalRotatedFiles;
  }
  async _deleteLastRotatedFile(rotatedFiles) {
    if (rotatedFiles.length < this.keepFiles) {
      return;
    }
    const lastFilePath = rotatedFiles.pop();
    await unlinkAsync(lastFilePath);
  }
  async _renameRotatedFilesByOne(rotatedFiles) {
    const logFileBaseName = (0, _path.basename)(this.logFilePath);
    const logFilesFolder = (0, _path.dirname)(this.logFilePath);
    for (let i = rotatedFiles.length - 1; i >= 0; i--) {
      const oldFilePath = rotatedFiles[i];
      const newFilePath = `${logFilesFolder}${_path.sep}${logFileBaseName}.${i + 1}`;
      await renameAsync(oldFilePath, newFilePath);
    }
  }
  async _rotateCurrentLogFile() {
    const newFilePath = `${this.logFilePath}.0`;
    await renameAsync(this.logFilePath, newFilePath);
  }
  _sendReloadLogConfigSignal() {
    if (!process.env.isDevCliChild || !process.send) {
      process.emit('SIGHUP', 'SIGHUP');
      return;
    }

    // Send a special message to the cluster manager
    // so it can forward it correctly
    // It will only run when we are under cluster mode (not under a production environment)
    process.send(['RELOAD_LOGGING_CONFIG_FROM_SERVER_WORKER']);
  }
}
exports.LogRotator = LogRotator;