"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BaseLogFormat = void 0;
var _stream = _interopRequireDefault(require("stream"));
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _lodash = _interopRequireDefault(require("lodash"));
var _queryString = _interopRequireDefault(require("query-string"));
var _numeral = _interopRequireDefault(require("@elastic/numeral"));
var _chalk = _interopRequireDefault(require("chalk"));
var _util = require("util");
var _utils = require("./utils");
var _metadata = require("./metadata");
var _log_events = require("./log_events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function serializeError(err = {}) {
  return {
    message: err.message,
    name: err.name,
    stack: err.stack,
    code: err.code,
    signal: err.signal
  };
}
const levelColor = function (code) {
  if (code < 299) return _chalk.default.green(String(code));
  if (code < 399) return _chalk.default.yellow(String(code));
  if (code < 499) return _chalk.default.magentaBright(String(code));
  return _chalk.default.red(String(code));
};
class BaseLogFormat extends _stream.default.Transform {
  constructor(config) {
    super({
      readableObjectMode: false,
      writableObjectMode: true
    });
    this.config = config;
  }
  filter(data) {
    if (!this.config.filter) {
      return data;
    }
    return (0, _utils.applyFiltersToKeys)(data, this.config.filter);
  }
  _transform(event, enc, next) {
    const data = this.filter(this.readEvent(event));
    this.push(this.format(data) + '\n');
    next();
  }
  getContentLength({
    responsePayload,
    responseHeaders
  }) {
    try {
      return (0, _utils.getResponsePayloadBytes)(responsePayload, responseHeaders);
    } catch (e) {
      // We intentionally swallow any errors as this information is
      // only a nicety for logging purposes, and should not cause the
      // server to crash if it cannot be determined.
      this.push(this.format({
        type: 'log',
        tags: ['warning', 'logging'],
        message: `Failed to calculate response payload bytes. [${e}]`
      }) + '\n');
    }
  }
  extractAndFormatTimestamp(data, format) {
    const {
      timezone
    } = this.config;
    const date = (0, _momentTimezone.default)(data['@timestamp']);
    if (timezone) {
      date.tz(timezone);
    }
    return date.format(format);
  }
  readEvent(event) {
    const data = {
      type: event.event,
      '@timestamp': event.timestamp,
      tags: [...(event.tags || [])],
      pid: event.pid
    };
    if ((0, _log_events.isResponseEvent)(event)) {
      _lodash.default.defaults(data, _lodash.default.pick(event, ['method', 'statusCode']));
      const source = _lodash.default.get(event, 'source', {});
      data.req = {
        url: event.path,
        method: event.method || '',
        headers: event.headers,
        remoteAddress: source.remoteAddress,
        userAgent: source.userAgent,
        referer: source.referer
      };
      data.res = {
        statusCode: event.statusCode,
        responseTime: event.responseTime,
        contentLength: this.getContentLength(event)
      };
      const query = _queryString.default.stringify(event.query, {
        sort: false
      });
      if (query) {
        data.req.url += '?' + query;
      }
      data.message = data.req.method.toUpperCase() + ' ';
      data.message += data.req.url;
      data.message += ' ';
      data.message += levelColor(data.res.statusCode);
      data.message += ' ';
      data.message += _chalk.default.gray(data.res.responseTime + 'ms');
      if (data.res.contentLength) {
        data.message += _chalk.default.gray(' - ' + (0, _numeral.default)(data.res.contentLength).format('0.0b'));
      }
    } else if ((0, _log_events.isOpsEvent)(event)) {
      _lodash.default.defaults(data, _lodash.default.pick(event, ['pid', 'os', 'proc', 'load']));
      data.message = _chalk.default.gray('memory: ');
      data.message += (0, _numeral.default)(_lodash.default.get(data, 'proc.mem.heapUsed')).format('0.0b');
      data.message += ' ';
      data.message += _chalk.default.gray('uptime: ');
      data.message += (0, _numeral.default)(_lodash.default.get(data, 'proc.uptime')).format('00:00:00');
      data.message += ' ';
      data.message += _chalk.default.gray('load: [');
      data.message += _lodash.default.get(data, 'os.load', []).map(val => {
        return (0, _numeral.default)(val).format('0.00');
      }).join(' ');
      data.message += _chalk.default.gray(']');
      data.message += ' ';
      data.message += _chalk.default.gray('delay: ');
      data.message += (0, _numeral.default)(_lodash.default.get(data, 'proc.delay')).format('0.000');
    } else if ((0, _log_events.isErrorEvent)(event)) {
      data.level = 'error';
      data.error = serializeError(event.error);
      data.url = event.url;
      const message = _lodash.default.get(event, 'error.message');
      data.message = message || 'Unknown error (no message)';
    } else if ((0, _log_events.isUndeclaredErrorEvent)(event)) {
      data.type = 'error';
      data.level = _lodash.default.includes(event.tags, 'fatal') ? 'fatal' : 'error';
      data.error = serializeError(event.error);
      const message = _lodash.default.get(event, 'error.message');
      data.message = message || 'Unknown error object (no message)';
    } else if ((0, _log_events.isLogEvent)(event)) {
      _lodash.default.assign(data, (0, _metadata.getLogEventData)(event.data));
    } else {
      data.message = _lodash.default.isString(event.data) ? event.data : (0, _util.inspect)(event.data);
    }
    return data;
  }
}
exports.BaseLogFormat = BaseLogFormat;