"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LegacyLoggingServer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _podium = _interopRequireDefault(require("@hapi/podium"));
var _setup_logging = require("./setup_logging");
var _metadata = require("./metadata");
var _schema = require("./schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const isEmptyObject = obj => Object.keys(obj).length === 0;
function getDataToLog(error, metadata, message) {
  if (error) {
    return error;
  }
  if (!isEmptyObject(metadata)) {
    return (0, _metadata.attachMetaData)(message, metadata);
  }
  return message;
}
/**
 * Converts core log level to a one that's known to the legacy platform.
 * @param level Log level from the core.
 */
function getLegacyLogLevel(level) {
  const logLevel = level.id.toLowerCase();
  if (logLevel === 'warn') {
    return 'warning';
  }
  if (logLevel === 'trace') {
    return 'debug';
  }
  return logLevel;
}

/**
 *  The "legacy" Kibana uses Hapi server + even-better plugin to log, so we should
 *  use the same approach here to make log records generated by the core to look the
 *  same as the rest of the records generated by the "legacy" Kibana. But to reduce
 *  overhead of having full blown Hapi server instance we create our own "light" version.
 *  @internal
 */
class LegacyLoggingServer {
  // Emulates Hapi's usage of the podium event bus.

  constructor(legacyLoggingConfig) {
    (0, _defineProperty2.default)(this, "connections", []);
    (0, _defineProperty2.default)(this, "events", new _podium.default(['log', 'request', 'response']));
    (0, _defineProperty2.default)(this, "onPostStopCallback", void 0);
    // We set `ops.interval` to max allowed number and `ops` filter to value
    // that doesn't exist to avoid logging of ops at all, if turned on it will be
    // logged by the "legacy" Kibana.
    const loggingConfig = _schema.legacyLoggingConfigSchema.validate({
      ...legacyLoggingConfig,
      events: {
        ...legacyLoggingConfig.events,
        ops: '__no-ops__'
      }
    });
    (0, _setup_logging.setupLogging)(this, loggingConfig, 2147483647);
  }
  register({
    plugin: {
      register
    },
    options
  }) {
    return register(this, options);
  }
  log({
    level,
    context,
    message,
    error,
    timestamp,
    meta = {}
  }) {
    const {
      tags = [],
      ...metadata
    } = meta;
    this.events.emit('log', {
      data: getDataToLog(error, metadata, message),
      tags: [getLegacyLogLevel(level), ...context.split('.'), ...tags],
      timestamp: timestamp.getTime()
    }).catch(err => {
      // eslint-disable-next-line no-console
      console.error('An unexpected error occurred while writing to the log:', err.stack);
      process.exit(1);
    });
  }
  stop() {
    // Tell the plugin we're stopping.
    if (this.onPostStopCallback !== undefined) {
      this.onPostStopCallback();
    }
  }
  ext(eventName, callback) {
    // method is called by plugin that's being registered.
    if (eventName === 'onPostStop') {
      this.onPostStopCallback = callback;
    }
    // We don't care about any others the plugin registers
  }

  expose() {
    // method is called by plugin that's being registered.
  }
}
exports.LegacyLoggingServer = LegacyLoggingServer;