import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
import _typeof from "@babel/runtime/helpers/typeof";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import { groupBy, has, isEqual } from 'lodash';
import { buildQueryFromKuery } from './from_kuery';
import { buildQueryFromFilters } from './from_filters';
import { buildQueryFromLucene } from './from_lucene';
function removeMatchAll(filters) {
  return filters.filter(function (filter) {
    return !filter || _typeof(filter) !== 'object' || !isEqual(filter, {
      match_all: {}
    });
  });
}

/**
 * @param indexPattern
 * @param queries - a query object or array of query objects. Each query has a language property and a query property.
 * @param filters - a filter object or array of filter objects
 * @param config - an objects with query:allowLeadingWildcards and query:queryString:options UI
 * settings in form of { allowLeadingWildcards, queryStringOptions }
 * config contains dateformat:tz
 *
 * @public
 */
export function buildEsQuery(indexPattern, queries, filters) {
  var config = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {
    allowLeadingWildcards: false,
    queryStringOptions: {},
    ignoreFilterIfFieldNotInIndex: false
  };
  queries = Array.isArray(queries) ? queries : [queries];
  filters = Array.isArray(filters) ? filters : [filters];
  var validQueries = queries.filter(function (query) {
    return has(query, 'query');
  });
  var queriesByLanguage = groupBy(validQueries, 'language');
  var kueryQuery = buildQueryFromKuery(indexPattern, queriesByLanguage.kuery, config.allowLeadingWildcards, config.dateFormatTZ, config.filtersInMustClause);
  var luceneQuery = buildQueryFromLucene(queriesByLanguage.lucene, config.queryStringOptions, config.dateFormatTZ);
  var filterQuery = buildQueryFromFilters(filters, indexPattern, config.ignoreFilterIfFieldNotInIndex);
  return {
    bool: {
      must: removeMatchAll([].concat(_toConsumableArray(kueryQuery.must), _toConsumableArray(luceneQuery.must), _toConsumableArray(filterQuery.must))),
      filter: removeMatchAll([].concat(_toConsumableArray(kueryQuery.filter), _toConsumableArray(luceneQuery.filter), _toConsumableArray(filterQuery.filter))),
      should: removeMatchAll([].concat(_toConsumableArray(kueryQuery.should), _toConsumableArray(luceneQuery.should), _toConsumableArray(filterQuery.should))),
      must_not: [].concat(_toConsumableArray(kueryQuery.must_not), _toConsumableArray(luceneQuery.must_not), _toConsumableArray(filterQuery.must_not))
    }
  };
}