import _typeof from "@babel/runtime/helpers/typeof";
import _objectSpread from "@babel/runtime/helpers/objectSpread2";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import { omit, get } from 'lodash';
import { FilterStateStore } from '../build_filters';

/**
 *
 * @param filter
 * @returns `true` if the filter should be applied to global scope
 *
 * @public
 */
export var isFilterPinned = function isFilterPinned(filter) {
  return filter.$state && filter.$state.store === FilterStateStore.GLOBAL_STATE;
};

/**
 * @param filter
 * @returns `true` if the filter is disabled
 *
 * @public
 */
export var isFilterDisabled = function isFilterDisabled(filter) {
  return get(filter, 'meta.disabled', false);
};

/**
 *
 * @param filter
 * @returns A copy of the filter with a toggled disabled state
 *
 * @public
 */
export var toggleFilterDisabled = function toggleFilterDisabled(filter) {
  var disabled = !filter.meta.disabled;
  var meta = _objectSpread(_objectSpread({}, filter.meta), {}, {
    disabled: disabled
  });
  return _objectSpread(_objectSpread({}, filter), {}, {
    meta: meta
  });
};

/**
 *
 * @param filter
 * @returns A copy of the filter with a toggled negated state
 *
 * @public
 */
export var toggleFilterNegated = function toggleFilterNegated(filter) {
  var negate = !filter.meta.negate;
  var meta = _objectSpread(_objectSpread({}, filter.meta), {}, {
    negate: negate
  });
  return _objectSpread(_objectSpread({}, filter), {}, {
    meta: meta
  });
};

/**
 *
 * @param filter
 * @returns A copy of the filter with a toggled pinned state (toggles store from app to global and vice versa)
 *
 * @public
 */
export var toggleFilterPinned = function toggleFilterPinned(filter) {
  var store = isFilterPinned(filter) ? FilterStateStore.APP_STATE : FilterStateStore.GLOBAL_STATE;
  var $state = _objectSpread(_objectSpread({}, filter.$state), {}, {
    store: store
  });
  return _objectSpread(_objectSpread({}, filter), {}, {
    $state: $state
  });
};

/**
 * @param filter
 * @returns An enabled copy of the filter
 *
 * @public
 */
export var enableFilter = function enableFilter(filter) {
  return !filter.meta.disabled ? filter : toggleFilterDisabled(filter);
};

/**
 * @param filter
 * @returns A disabled copy of the filter
 *
 * @public
 */
export var disableFilter = function disableFilter(filter) {
  return filter.meta.disabled ? filter : toggleFilterDisabled(filter);
};

/**
 * @param filter
 * @returns A pinned (global) copy of the filter
 *
 * @public
 */
export var pinFilter = function pinFilter(filter) {
  return isFilterPinned(filter) ? filter : toggleFilterPinned(filter);
};

/**
 * @param filter
 * @returns An unpinned (app scoped) copy of the filter
 *
 * @public
 */
export var unpinFilter = function unpinFilter(filter) {
  return !isFilterPinned(filter) ? filter : toggleFilterPinned(filter);
};

/**
 * @param {unknown} filter
 * @returns `true` if the given object is a filter
 *
 * @public
 */
export var isFilter = function isFilter(x) {
  return !!x && _typeof(x) === 'object' && !!x.meta && _typeof(x.meta) === 'object' && typeof x.meta.disabled === 'boolean';
};

/**
 * @param {unknown} filters
 * @returns `true` if the given object is an array of filters
 *
 * @public
 */
export var isFilters = function isFilters(x) {
  return Array.isArray(x) && !x.find(function (y) {
    return !isFilter(y);
  });
};

/**
 * Clean out decorators from the filters
 * @param {object} filter The filter to clean
 * @returns {object}
 *
 * @public
 */
export var cleanFilter = function cleanFilter(filter) {
  return omit(filter, ['meta', '$state']);
};