"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ConfigService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _config = require("./config");
var _deprecation = require("./deprecation");
var _legacy = require("./legacy");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/** @internal */
class ConfigService {
  /**
   * Whenever a config if read at a path, we mark that path as 'handled'. We can
   * then list all unhandled config paths when the startup process is completed.
   */

  constructor(rawConfigProvider, env, logger) {
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "deprecationLog", void 0);
    (0, _defineProperty2.default)(this, "validated", false);
    (0, _defineProperty2.default)(this, "config$", void 0);
    (0, _defineProperty2.default)(this, "lastConfig", void 0);
    (0, _defineProperty2.default)(this, "deprecatedConfigPaths", new _rxjs.BehaviorSubject({
      set: [],
      unset: []
    }));
    (0, _defineProperty2.default)(this, "handledPaths", new Set());
    (0, _defineProperty2.default)(this, "schemas", new Map());
    (0, _defineProperty2.default)(this, "deprecations", new _rxjs.BehaviorSubject([]));
    (0, _defineProperty2.default)(this, "handledDeprecatedConfigs", new Map());
    this.rawConfigProvider = rawConfigProvider;
    this.env = env;
    this.log = logger.get('config');
    this.deprecationLog = logger.get('config', 'deprecation');
    this.config$ = (0, _rxjs.combineLatest)([this.rawConfigProvider.getConfig$(), this.deprecations]).pipe((0, _operators.map)(([rawConfig, deprecations]) => {
      const migrated = (0, _deprecation.applyDeprecations)(rawConfig, deprecations);
      this.deprecatedConfigPaths.next(migrated.changedPaths);
      return new _legacy.LegacyObjectToConfigAdapter(migrated.config);
    }), (0, _operators.tap)(config => {
      this.lastConfig = config;
    }), (0, _operators.shareReplay)(1));
  }

  /**
   * Set config schema for a path and performs its validation
   */
  setSchema(path, schema) {
    const namespace = pathToString(path);
    if (this.schemas.has(namespace)) {
      throw new Error(`Validation schema for [${path}] was already registered.`);
    }
    this.schemas.set(namespace, schema);
    this.markAsHandled(path);
  }

  /**
   * Register a {@link ConfigDeprecationProvider} to be used when validating and migrating the configuration
   */
  addDeprecationProvider(path, provider) {
    const flatPath = pathToString(path);
    this.deprecations.next([...this.deprecations.value, ...provider(_deprecation.configDeprecationFactory).map(deprecation => ({
      deprecation,
      path: flatPath,
      context: createDeprecationContext(this.env)
    }))]);
  }

  /**
   * returns all handled deprecated configs
   */
  getHandledDeprecatedConfigs() {
    return [...this.handledDeprecatedConfigs.entries()];
  }

  /**
   * Validate the whole configuration and log the deprecation warnings.
   *
   * This must be done after every schemas and deprecation providers have been registered.
   */
  async validate(params = {
    logDeprecations: true
  }) {
    const namespaces = [...this.schemas.keys()];
    for (let i = 0; i < namespaces.length; i++) {
      await this.getValidatedConfigAtPath$(namespaces[i]).pipe((0, _operators.first)()).toPromise();
    }
    if (params.logDeprecations) {
      await this.logDeprecation();
    }
    this.validated = true;
  }

  /**
   * Returns the full config object observable. This is not intended for
   * "normal use", but for internal features that _need_ access to the full object.
   */
  getConfig$() {
    return this.config$;
  }

  /**
   * Reads the subset of the config at the specified `path` and validates it
   * against its registered schema.
   *
   * @param path - The path to the desired subset of the config.
   */
  atPath(path) {
    return this.getValidatedConfigAtPath$(path);
  }

  /**
   * Similar to {@link atPath}, but return the last emitted value synchronously instead of an
   * observable.
   *
   * @param path - The path to the desired subset of the config.
   */
  atPathSync(path) {
    if (!this.validated) {
      throw new Error('`atPathSync` called before config was validated');
    }
    const configAtPath = this.lastConfig.get(path);
    return this.validateAtPath(path, configAtPath);
  }
  async isEnabledAtPath(path) {
    var _validatedConfig$enab;
    const namespace = pathToString(path);
    const validatedConfig = this.schemas.has(namespace) ? await this.atPath(path).pipe((0, _operators.first)()).toPromise() : undefined;
    const enabledPath = createPluginEnabledPath(path);
    const config = await this.config$.pipe((0, _operators.first)()).toPromise();

    // if plugin hasn't got a config schema, we try to read "enabled" directly
    const isEnabled = (_validatedConfig$enab = validatedConfig === null || validatedConfig === void 0 ? void 0 : validatedConfig.enabled) !== null && _validatedConfig$enab !== void 0 ? _validatedConfig$enab : config.get(enabledPath);

    // if we implicitly added an `enabled` config to a plugin without a schema,
    // we log a deprecation warning, as this will not be supported in 8.0
    if ((validatedConfig === null || validatedConfig === void 0 ? void 0 : validatedConfig.enabled) === undefined && isEnabled !== undefined) {
      const deprecationPath = pathToString(enabledPath);
      const deprecatedConfigDetails = {
        configPath: deprecationPath,
        title: `Setting "${deprecationPath}" is deprecated`,
        message: `Configuring "${deprecationPath}" is deprecated and will be removed in 8.0.0.`,
        correctiveActions: {
          manualSteps: [`Remove "${deprecationPath}" from the Kibana config file, CLI flag, or environment variable (in Docker only) before upgrading to 8.0.0.`]
        }
      };
      this.deprecationLog.warn(deprecatedConfigDetails.message);
      this.markDeprecatedConfigAsHandled(namespace, deprecatedConfigDetails);
    }

    // not declared. consider that plugin is enabled by default
    if (isEnabled === undefined) {
      return true;
    }
    if (isEnabled === false) {
      // If the plugin is _not_ enabled, we mark the entire plugin path as
      // handled, as it's expected that it won't be used.
      this.markAsHandled(path);
      return false;
    }

    // If plugin enabled we mark the enabled path as handled, as we for example
    // can have plugins that don't have _any_ config except for this field, and
    // therefore have no reason to try to get the config.
    this.markAsHandled(enabledPath);
    return true;
  }
  async getUnusedPaths() {
    const config = await this.config$.pipe((0, _operators.first)()).toPromise();
    const handledPaths = [...this.handledPaths.values()].map(pathToString);
    return config.getFlattenedPaths().filter(path => !isPathHandled(path, handledPaths));
  }
  async getUsedPaths() {
    const config = await this.config$.pipe((0, _operators.first)()).toPromise();
    const handledPaths = [...this.handledPaths.values()].map(pathToString);
    return config.getFlattenedPaths().filter(path => isPathHandled(path, handledPaths));
  }
  getDeprecatedConfigPath$() {
    return this.deprecatedConfigPaths.asObservable();
  }
  async logDeprecation() {
    const rawConfig = await this.rawConfigProvider.getConfig$().pipe((0, _operators.take)(1)).toPromise();
    const deprecations = await this.deprecations.pipe((0, _operators.take)(1)).toPromise();
    const deprecationMessages = [];
    const createAddDeprecation = domainId => context => {
      if (!context.silent) {
        deprecationMessages.push(context.message);
      }
      this.markDeprecatedConfigAsHandled(domainId, context);
    };
    (0, _deprecation.applyDeprecations)(rawConfig, deprecations, createAddDeprecation);
    deprecationMessages.forEach(msg => {
      this.deprecationLog.warn(msg);
    });
  }
  validateAtPath(path, config) {
    const namespace = pathToString(path);
    const schema = this.schemas.get(namespace);
    if (!schema) {
      throw new Error(`No validation schema has been defined for [${namespace}]`);
    }
    return schema.validate(config, {
      dev: this.env.mode.dev,
      prod: this.env.mode.prod,
      ...this.env.packageInfo
    }, `config validation of [${namespace}]`);
  }
  getValidatedConfigAtPath$(path) {
    return this.config$.pipe((0, _operators.map)(config => config.get(path)), (0, _operators.distinctUntilChanged)(_lodash.isEqual), (0, _operators.map)(config => this.validateAtPath(path, config)));
  }
  markAsHandled(path) {
    this.log.debug(`Marking config path as handled: ${path}`);
    this.handledPaths.add(path);
  }
  markDeprecatedConfigAsHandled(domainId, config) {
    const handledDeprecatedConfig = this.handledDeprecatedConfigs.get(domainId) || [];
    handledDeprecatedConfig.push(config);
    this.handledDeprecatedConfigs.set(domainId, handledDeprecatedConfig);
  }
}
exports.ConfigService = ConfigService;
const createPluginEnabledPath = configPath => {
  if (Array.isArray(configPath)) {
    return configPath.concat('enabled');
  }
  return `${configPath}.enabled`;
};
const pathToString = path => Array.isArray(path) ? path.join('.') : path;

/**
 * A path is considered 'handled' if it is a subset of any of the already
 * handled paths.
 */
const isPathHandled = (path, handledPaths) => handledPaths.some(handledPath => (0, _config.hasConfigPathIntersection)(path, handledPath));
const createDeprecationContext = env => {
  return {
    branch: env.packageInfo.branch,
    version: env.packageInfo.version
  };
};