import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
import _objectSpread from "@babel/runtime/helpers/objectSpread2";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { isLeft } from 'fp-ts/lib/Either';
import { PathReporter } from 'io-ts/lib/PathReporter';
import { matchRoutes as matchRoutesConfig } from 'react-router-config';
import qs from 'query-string';
import { findLastIndex, merge, compact } from 'lodash';
import { mergeRt } from '@kbn/io-ts-utils/merge_rt';
import { deepExactRt } from '@kbn/io-ts-utils/deep_exact_rt';

function toReactRouterPath(path) {
  return path.replace(/(?:{([^\/]+)})/g, ':$1');
}

export function createRouter(routes) {
  var routesByReactRouterConfig = new Map();
  var reactRouterConfigsByRoute = new Map();
  var reactRouterConfigs = routes.map(function (route) {
    return toReactRouterConfigRoute(route);
  });

  function toReactRouterConfigRoute(route) {
    var _map, _route$children, _route$children2;

    var reactRouterConfig = {
      component: function component() {
        return route.element;
      },
      routes: (_map = (_route$children = route.children) === null || _route$children === void 0 ? void 0 : _route$children.map(function (child) {
        return toReactRouterConfigRoute(child);
      })) !== null && _map !== void 0 ? _map : [],
      exact: !((_route$children2 = route.children) !== null && _route$children2 !== void 0 && _route$children2.length),
      path: toReactRouterPath(route.path)
    };
    routesByReactRouterConfig.set(reactRouterConfig, route);
    reactRouterConfigsByRoute.set(route, reactRouterConfig);
    return reactRouterConfig;
  }

  function _getRoutesToMatch(path) {
    var matches = matchRoutesConfig(reactRouterConfigs, toReactRouterPath(path));

    if (!matches.length) {
      throw new Error("No matching route found for ".concat(path));
    }

    var matchedRoutes = matches.map(function (match) {
      return routesByReactRouterConfig.get(match.route);
    });
    return matchedRoutes;
  }

  var _matchRoutes = function matchRoutes() {
    var optional = false;

    for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
      args[_key] = arguments[_key];
    }

    if (typeof args[args.length - 1] === 'boolean') {
      optional = args[args.length - 1];
      args.pop();
    }

    var location = args[args.length - 1];
    args.pop();
    var paths = args;

    if (paths.length === 0) {
      paths = [location.pathname || '/'];
    }

    var matches = [];
    var matchIndex = -1;

    var _loop = function _loop() {
      var path = _paths[_i];
      var greedy = path.endsWith('/*') || args.length === 0;
      matches = matchRoutesConfig(reactRouterConfigs, toReactRouterPath(location.pathname));
      matchIndex = greedy ? matches.length - 1 : findLastIndex(matches, function (match) {
        return match.route.path === toReactRouterPath(path);
      });

      if (matchIndex !== -1) {
        return "break";
      }

      matchIndex = -1;
    };

    for (var _i = 0, _paths = paths; _i < _paths.length; _i++) {
      var _ret = _loop();

      if (_ret === "break") break;
    }

    if (matchIndex === -1) {
      if (optional) {
        return [];
      }

      throw new Error("No matching route found for ".concat(paths));
    }

    return matches.slice(0, matchIndex + 1).map(function (matchedRoute) {
      var route = routesByReactRouterConfig.get(matchedRoute.route);

      if (route !== null && route !== void 0 && route.params) {
        var _route$defaults;

        var decoded = deepExactRt(route.params).decode(merge({}, (_route$defaults = route.defaults) !== null && _route$defaults !== void 0 ? _route$defaults : {}, {
          path: matchedRoute.match.params,
          query: qs.parse(location.search, {
            decode: true
          })
        }));

        if (isLeft(decoded)) {
          throw new Error(PathReporter.report(decoded).join('\n'));
        }

        return {
          match: _objectSpread(_objectSpread({}, matchedRoute.match), {}, {
            params: decoded.right
          }),
          route: route
        };
      }

      return {
        match: _objectSpread(_objectSpread({}, matchedRoute.match), {}, {
          params: {
            path: {},
            query: {}
          }
        }),
        route: route
      };
    });
  };

  var _link = function link(path) {
    var params = arguments.length <= 1 ? undefined : arguments[1];
    var paramsWithBuiltInDefaults = merge({
      path: {},
      query: {}
    }, params);
    path = path.split('/').map(function (part) {
      var match = part.match(/(?:{([a-zA-Z]+)})/);
      return match ? paramsWithBuiltInDefaults.path[match[1]] : part;
    }).join('/');

    var matchedRoutes = _getRoutesToMatch(path);

    var validationType = mergeRt.apply(void 0, _toConsumableArray(compact(matchedRoutes.map(function (match) {
      return match.params;
    }))));
    var paramsWithRouteDefaults = merge.apply(void 0, [{}].concat(_toConsumableArray(matchedRoutes.map(function (route) {
      var _route$defaults2;

      return (_route$defaults2 = route.defaults) !== null && _route$defaults2 !== void 0 ? _route$defaults2 : {};
    })), [paramsWithBuiltInDefaults]));
    var validation = validationType.decode(paramsWithRouteDefaults);

    if (isLeft(validation)) {
      throw new Error(PathReporter.report(validation).join('\n'));
    }

    return qs.stringifyUrl({
      url: path,
      query: paramsWithRouteDefaults.query
    }, {
      encode: true
    });
  };

  return {
    link: function link(path) {
      for (var _len2 = arguments.length, args = new Array(_len2 > 1 ? _len2 - 1 : 0), _key2 = 1; _key2 < _len2; _key2++) {
        args[_key2 - 1] = arguments[_key2];
      }

      return _link.apply(void 0, [path].concat(args));
    },
    getParams: function getParams() {
      var matches = _matchRoutes.apply(void 0, arguments);

      return matches.length ? merge.apply(void 0, [{
        path: {},
        query: {}
      }].concat(_toConsumableArray(matches.map(function (match) {
        var _match$route$defaults, _match$route;

        return merge({}, (_match$route$defaults = (_match$route = match.route) === null || _match$route === void 0 ? void 0 : _match$route.defaults) !== null && _match$route$defaults !== void 0 ? _match$route$defaults : {}, match.match.params);
      })))) : undefined;
    },
    matchRoutes: function matchRoutes() {
      return _matchRoutes.apply(void 0, arguments);
    },
    getRoutePath: function getRoutePath(route) {
      return reactRouterConfigsByRoute.get(route).path;
    },
    getRoutesToMatch: function getRoutesToMatch(path) {
      return _getRoutesToMatch(path);
    }
  };
}