/**
 * This helper creates a live region that announces the results of certain
 * actions (e.g. searching, paging, etc.), that are otherwise invisible
 * to screen reader users.
 *
 * @see https://developer.mozilla.org/en-US/docs/Web/Accessibility/ARIA/ARIA_Live_Regions
 */
var regionId = "search-ui-screen-reader-notifications";
var hasDOM = typeof document !== "undefined"; // Prevent errors in SSR apps

var getLiveRegion = function getLiveRegion() {
  if (!hasDOM) return;
  var region = document.getElementById(regionId);
  if (region) return region;
  region = document.createElement("div");
  region.id = regionId;
  region.setAttribute("role", "status");
  region.setAttribute("aria-live", "polite");
  /**
   * Visually-hidden CSS that's still available to screen readers.
   * We're avoiding putting this in a stylesheet to ensure that this
   * still works for users that opt for custom views & CSS. We're
   * also opting to use CSSOM instead of inline styles to avoid
   * Content Security Policy warnings.
   *
   * @see https://accessibility.18f.gov/hidden-content/
   */

  region.style.position = "absolute";
  region.style.width = "1px";
  region.style.height = "1px";
  region.style.margin = "-1px";
  region.style.padding = "0";
  region.style.border = "0";
  region.style.overflow = "hidden";
  region.style.clip = "rect(0 0 0 0)";
  document.body.appendChild(region);
  return region;
};

var announceToScreenReader = function announceToScreenReader(announcement) {
  if (hasDOM) {
    var region = getLiveRegion();
    region.textContent = announcement;
  }
};

var defaultMessages = {
  searchResults: function searchResults(_ref) {
    var start = _ref.start,
        end = _ref.end,
        totalResults = _ref.totalResults,
        searchTerm = _ref.searchTerm;
    var message = "Showing ".concat(start, " to ").concat(end, " results out of ").concat(totalResults);
    if (searchTerm) message += ", searching for \"".concat(searchTerm, "\".");
    return message;
  }
};
export { getLiveRegion, announceToScreenReader, defaultMessages };
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uL3NyYy9BMTF5Tm90aWZpY2F0aW9ucy5qcyJdLCJuYW1lcyI6WyJyZWdpb25JZCIsImhhc0RPTSIsImRvY3VtZW50IiwiZ2V0TGl2ZVJlZ2lvbiIsInJlZ2lvbiIsImdldEVsZW1lbnRCeUlkIiwiY3JlYXRlRWxlbWVudCIsImlkIiwic2V0QXR0cmlidXRlIiwic3R5bGUiLCJwb3NpdGlvbiIsIndpZHRoIiwiaGVpZ2h0IiwibWFyZ2luIiwicGFkZGluZyIsImJvcmRlciIsIm92ZXJmbG93IiwiY2xpcCIsImJvZHkiLCJhcHBlbmRDaGlsZCIsImFubm91bmNlVG9TY3JlZW5SZWFkZXIiLCJhbm5vdW5jZW1lbnQiLCJ0ZXh0Q29udGVudCIsImRlZmF1bHRNZXNzYWdlcyIsInNlYXJjaFJlc3VsdHMiLCJzdGFydCIsImVuZCIsInRvdGFsUmVzdWx0cyIsInNlYXJjaFRlcm0iLCJtZXNzYWdlIl0sIm1hcHBpbmdzIjoiQUFBQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLElBQU1BLFFBQVEsR0FBRyx1Q0FBakI7QUFDQSxJQUFNQyxNQUFNLEdBQUcsT0FBT0MsUUFBUCxLQUFvQixXQUFuQyxDLENBQWdEOztBQUVoRCxJQUFNQyxhQUFhLEdBQUcsU0FBaEJBLGFBQWdCLEdBQU07QUFDMUIsTUFBSSxDQUFDRixNQUFMLEVBQWE7QUFFYixNQUFJRyxNQUFNLEdBQUdGLFFBQVEsQ0FBQ0csY0FBVCxDQUF3QkwsUUFBeEIsQ0FBYjtBQUNBLE1BQUlJLE1BQUosRUFBWSxPQUFPQSxNQUFQO0FBRVpBLEVBQUFBLE1BQU0sR0FBR0YsUUFBUSxDQUFDSSxhQUFULENBQXVCLEtBQXZCLENBQVQ7QUFDQUYsRUFBQUEsTUFBTSxDQUFDRyxFQUFQLEdBQVlQLFFBQVo7QUFDQUksRUFBQUEsTUFBTSxDQUFDSSxZQUFQLENBQW9CLE1BQXBCLEVBQTRCLFFBQTVCO0FBQ0FKLEVBQUFBLE1BQU0sQ0FBQ0ksWUFBUCxDQUFvQixXQUFwQixFQUFpQyxRQUFqQztBQUVBO0FBQ0Y7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFDRUosRUFBQUEsTUFBTSxDQUFDSyxLQUFQLENBQWFDLFFBQWIsR0FBd0IsVUFBeEI7QUFDQU4sRUFBQUEsTUFBTSxDQUFDSyxLQUFQLENBQWFFLEtBQWIsR0FBcUIsS0FBckI7QUFDQVAsRUFBQUEsTUFBTSxDQUFDSyxLQUFQLENBQWFHLE1BQWIsR0FBc0IsS0FBdEI7QUFDQVIsRUFBQUEsTUFBTSxDQUFDSyxLQUFQLENBQWFJLE1BQWIsR0FBc0IsTUFBdEI7QUFDQVQsRUFBQUEsTUFBTSxDQUFDSyxLQUFQLENBQWFLLE9BQWIsR0FBdUIsR0FBdkI7QUFDQVYsRUFBQUEsTUFBTSxDQUFDSyxLQUFQLENBQWFNLE1BQWIsR0FBc0IsR0FBdEI7QUFDQVgsRUFBQUEsTUFBTSxDQUFDSyxLQUFQLENBQWFPLFFBQWIsR0FBd0IsUUFBeEI7QUFDQVosRUFBQUEsTUFBTSxDQUFDSyxLQUFQLENBQWFRLElBQWIsR0FBb0IsZUFBcEI7QUFFQWYsRUFBQUEsUUFBUSxDQUFDZ0IsSUFBVCxDQUFjQyxXQUFkLENBQTBCZixNQUExQjtBQUNBLFNBQU9BLE1BQVA7QUFDRCxDQS9CRDs7QUFpQ0EsSUFBTWdCLHNCQUFzQixHQUFHLFNBQXpCQSxzQkFBeUIsQ0FBQUMsWUFBWSxFQUFJO0FBQzdDLE1BQUlwQixNQUFKLEVBQVk7QUFDVixRQUFNRyxNQUFNLEdBQUdELGFBQWEsRUFBNUI7QUFDQUMsSUFBQUEsTUFBTSxDQUFDa0IsV0FBUCxHQUFxQkQsWUFBckI7QUFDRDtBQUNGLENBTEQ7O0FBT0EsSUFBTUUsZUFBZSxHQUFHO0FBQ3RCQyxFQUFBQSxhQUFhLEVBQUUsNkJBQThDO0FBQUEsUUFBM0NDLEtBQTJDLFFBQTNDQSxLQUEyQztBQUFBLFFBQXBDQyxHQUFvQyxRQUFwQ0EsR0FBb0M7QUFBQSxRQUEvQkMsWUFBK0IsUUFBL0JBLFlBQStCO0FBQUEsUUFBakJDLFVBQWlCLFFBQWpCQSxVQUFpQjtBQUMzRCxRQUFJQyxPQUFPLHFCQUFjSixLQUFkLGlCQUEwQkMsR0FBMUIsNkJBQWdEQyxZQUFoRCxDQUFYO0FBQ0EsUUFBSUMsVUFBSixFQUFnQkMsT0FBTyxnQ0FBd0JELFVBQXhCLFFBQVA7QUFDaEIsV0FBT0MsT0FBUDtBQUNEO0FBTHFCLENBQXhCO0FBUUEsU0FBUzFCLGFBQVQsRUFBd0JpQixzQkFBeEIsRUFBZ0RHLGVBQWhEIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBUaGlzIGhlbHBlciBjcmVhdGVzIGEgbGl2ZSByZWdpb24gdGhhdCBhbm5vdW5jZXMgdGhlIHJlc3VsdHMgb2YgY2VydGFpblxuICogYWN0aW9ucyAoZS5nLiBzZWFyY2hpbmcsIHBhZ2luZywgZXRjLiksIHRoYXQgYXJlIG90aGVyd2lzZSBpbnZpc2libGVcbiAqIHRvIHNjcmVlbiByZWFkZXIgdXNlcnMuXG4gKlxuICogQHNlZSBodHRwczovL2RldmVsb3Blci5tb3ppbGxhLm9yZy9lbi1VUy9kb2NzL1dlYi9BY2Nlc3NpYmlsaXR5L0FSSUEvQVJJQV9MaXZlX1JlZ2lvbnNcbiAqL1xuY29uc3QgcmVnaW9uSWQgPSBcInNlYXJjaC11aS1zY3JlZW4tcmVhZGVyLW5vdGlmaWNhdGlvbnNcIjtcbmNvbnN0IGhhc0RPTSA9IHR5cGVvZiBkb2N1bWVudCAhPT0gXCJ1bmRlZmluZWRcIjsgLy8gUHJldmVudCBlcnJvcnMgaW4gU1NSIGFwcHNcblxuY29uc3QgZ2V0TGl2ZVJlZ2lvbiA9ICgpID0+IHtcbiAgaWYgKCFoYXNET00pIHJldHVybjtcblxuICBsZXQgcmVnaW9uID0gZG9jdW1lbnQuZ2V0RWxlbWVudEJ5SWQocmVnaW9uSWQpO1xuICBpZiAocmVnaW9uKSByZXR1cm4gcmVnaW9uO1xuXG4gIHJlZ2lvbiA9IGRvY3VtZW50LmNyZWF0ZUVsZW1lbnQoXCJkaXZcIik7XG4gIHJlZ2lvbi5pZCA9IHJlZ2lvbklkO1xuICByZWdpb24uc2V0QXR0cmlidXRlKFwicm9sZVwiLCBcInN0YXR1c1wiKTtcbiAgcmVnaW9uLnNldEF0dHJpYnV0ZShcImFyaWEtbGl2ZVwiLCBcInBvbGl0ZVwiKTtcblxuICAvKipcbiAgICogVmlzdWFsbHktaGlkZGVuIENTUyB0aGF0J3Mgc3RpbGwgYXZhaWxhYmxlIHRvIHNjcmVlbiByZWFkZXJzLlxuICAgKiBXZSdyZSBhdm9pZGluZyBwdXR0aW5nIHRoaXMgaW4gYSBzdHlsZXNoZWV0IHRvIGVuc3VyZSB0aGF0IHRoaXNcbiAgICogc3RpbGwgd29ya3MgZm9yIHVzZXJzIHRoYXQgb3B0IGZvciBjdXN0b20gdmlld3MgJiBDU1MuIFdlJ3JlXG4gICAqIGFsc28gb3B0aW5nIHRvIHVzZSBDU1NPTSBpbnN0ZWFkIG9mIGlubGluZSBzdHlsZXMgdG8gYXZvaWRcbiAgICogQ29udGVudCBTZWN1cml0eSBQb2xpY3kgd2FybmluZ3MuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9hY2Nlc3NpYmlsaXR5LjE4Zi5nb3YvaGlkZGVuLWNvbnRlbnQvXG4gICAqL1xuICByZWdpb24uc3R5bGUucG9zaXRpb24gPSBcImFic29sdXRlXCI7XG4gIHJlZ2lvbi5zdHlsZS53aWR0aCA9IFwiMXB4XCI7XG4gIHJlZ2lvbi5zdHlsZS5oZWlnaHQgPSBcIjFweFwiO1xuICByZWdpb24uc3R5bGUubWFyZ2luID0gXCItMXB4XCI7XG4gIHJlZ2lvbi5zdHlsZS5wYWRkaW5nID0gXCIwXCI7XG4gIHJlZ2lvbi5zdHlsZS5ib3JkZXIgPSBcIjBcIjtcbiAgcmVnaW9uLnN0eWxlLm92ZXJmbG93ID0gXCJoaWRkZW5cIjtcbiAgcmVnaW9uLnN0eWxlLmNsaXAgPSBcInJlY3QoMCAwIDAgMClcIjtcblxuICBkb2N1bWVudC5ib2R5LmFwcGVuZENoaWxkKHJlZ2lvbik7XG4gIHJldHVybiByZWdpb247XG59O1xuXG5jb25zdCBhbm5vdW5jZVRvU2NyZWVuUmVhZGVyID0gYW5ub3VuY2VtZW50ID0+IHtcbiAgaWYgKGhhc0RPTSkge1xuICAgIGNvbnN0IHJlZ2lvbiA9IGdldExpdmVSZWdpb24oKTtcbiAgICByZWdpb24udGV4dENvbnRlbnQgPSBhbm5vdW5jZW1lbnQ7XG4gIH1cbn07XG5cbmNvbnN0IGRlZmF1bHRNZXNzYWdlcyA9IHtcbiAgc2VhcmNoUmVzdWx0czogKHsgc3RhcnQsIGVuZCwgdG90YWxSZXN1bHRzLCBzZWFyY2hUZXJtIH0pID0+IHtcbiAgICBsZXQgbWVzc2FnZSA9IGBTaG93aW5nICR7c3RhcnR9IHRvICR7ZW5kfSByZXN1bHRzIG91dCBvZiAke3RvdGFsUmVzdWx0c31gO1xuICAgIGlmIChzZWFyY2hUZXJtKSBtZXNzYWdlICs9IGAsIHNlYXJjaGluZyBmb3IgXCIke3NlYXJjaFRlcm19XCIuYDtcbiAgICByZXR1cm4gbWVzc2FnZTtcbiAgfVxufTtcblxuZXhwb3J0IHsgZ2V0TGl2ZVJlZ2lvbiwgYW5ub3VuY2VUb1NjcmVlblJlYWRlciwgZGVmYXVsdE1lc3NhZ2VzIH07XG4iXX0=